<?php

namespace Vtlabs\Smartstudy\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Smartstudy\Models\Subject;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Smartstudy\Filters\SubjectFilter;
use Vtlabs\Smartstudy\Http\Resources\Admin\SubjectAdminResource;

class SubjectController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'school' => 'sometimes|exists:smartstudy_schools,id'
        ]);

        $subjects = Subject::filter($request->all(), SubjectFilter::class);

        if ($request->pagination == '0') {
            $subjects = $subjects->get();
        } else {
            $subjects = $subjects->paginate();
        }

        return SubjectAdminResource::collection($subjects);
    }

    public function show(Subject $subject)
    {
        return new SubjectAdminResource($subject);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'faculty_id' => 'required|exists:smartstudy_faculties,id',
            'meta' => 'sometimes|json|nullable'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        $subject = Subject::create($request->all());

        return new SubjectAdminResource($subject);
    }

    public function update(Subject $subject, Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'faculty_id' => 'required|exists:smartstudy_faculties,id',
            'meta' => 'sometimes|json|nullable'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        $subject->fill($request->all());
        $subject->save();

        return new SubjectAdminResource($subject);
    }

    public function destroy(Subject $subject)
    {
        Gate::authorize('delete');

        $subject->delete();

        return response()->json([], 204);
    }
}
