<?php

namespace Vtlabs\Smartstudy\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Smartstudy\Models\Student;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Smartstudy\Filters\StudentFilter;
use Vtlabs\Smartstudy\Http\Resources\Admin\StudentAdminResource;

class StudentController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'school' => 'sometimes|exists:smartstudy_schools,id'
        ]);

        $students = Student::filter($request->all(), StudentFilter::class);

        if ($request->pagination == '0') {
            $students = $students->get();
        } else {
            $students = $students->paginate();
        }

        return StudentAdminResource::collection($students);
    }

    public function show(Student $student)
    {
        return new StudentAdminResource($student);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'meta' => 'sometimes|json|nullable'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        // create user for student first  
        $userData = collect($request->only(['name', 'email', 'mobile_number', 'password']));
        $userData = $userData->merge(['mobile_verified' => 1, 'roles' => ['student']]);
        $user = UserService::create($userData);

        $student = Student::where('user_id', $user->id)->firstOrFail();
        $student = $student->fill($request->only(['school_id', 'class_id', 'meta']));
        $student->save();

        return new StudentAdminResource($student);
    }

    public function update(Student $student, Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'meta' => 'sometimes|json|nullable'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        $student->fill($request->only(['school_id', 'class_id', 'meta']));
        $student->save();

        return new StudentAdminResource($student);
    }

    public function destroy(Student $student)
    {
        Gate::authorize('delete');

        $student->delete();

        $student->user->delete();

        return response()->json([], 204);
    }
}
