<?php

namespace Vtlabs\Smartstudy\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Smartstudy\Models\Session;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Smartstudy\Filters\SessionFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Smartstudy\Http\Resources\Admin\SessionAdminResource;

class SessionController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'school' => 'sometimes|exists:smartstudy_schools,id'
        ]);

        $sessions = Session::filter($request->all(), SessionFilter::class);

        if ($request->pagination == '0') {
            $sessions = $sessions->get();
        } else {
            $sessions = $sessions->paginate();
        }

        return SessionAdminResource::collection($sessions);
    }

    public function show(Session $session)
    {
        return new SessionAdminResource($session);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'subject_id' => 'required|exists:smartstudy_subjects,id',
            'faculty_id' => 'required|exists:smartstudy_faculties,id',
            'session_date' => 'required',
            'duration_minutes' => 'required|numeric',
            'meta' => 'sometimes|json|nullable'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        $session = Session::create($request->only([
            'title', 'description', 'school_id', 'class_id', 'subject_id', 'session_date', 'duration_minutes', 'meta',
            'faculty_id'
        ]));

        return new SessionAdminResource($session);
    }

    public function update(Session $session, Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'subject_id' => 'required|exists:smartstudy_subjects,id',
            'faculty_id' => 'required|exists:smartstudy_faculties,id',
            'session_date' => 'required',
            'duration_minutes' => 'required|numeric',
            'meta' => 'sometimes|json|nullable'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        $session->fill($request->only([
            'title', 'description', 'school_id', 'class_id', 'subject_id', 'session_date', 'duration_minutes', 'meta',
            'faculty_id'
        ]));
        $session->save();

        return new SessionAdminResource($session);
    }

    public function destroy(Session $session)
    {
        Gate::authorize('delete');

        $session->delete();

        return response()->json([], 204);
    }
}
