<?php

namespace Vtlabs\Smartstudy\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Smartstudy\Models\Faculty;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Smartstudy\Filters\FacultyFilter;
use Vtlabs\Smartstudy\Http\Resources\Admin\FacultyAdminResource;

class FacultyController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'school' => 'sometimes|exists:smartstudy_schools,id'
        ]);

        $faculty = Faculty::filter($request->all(), FacultyFilter::class);

        if ($request->pagination == '0') {
            $faculty = $faculty->get();
        } else {
            $faculty = $faculty->paginate();
        }

        return FacultyAdminResource::collection($faculty);
    }

    public function show(Faculty $faculty)
    {
        return new FacultyAdminResource($faculty);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'meta' => 'sometimes|json|nullable',
            'description' => 'sometimes'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        // create user for faculty first  
        $userData = collect($request->only(['name', 'email', 'mobile_number', 'password']));
        $userData = $userData->merge(['mobile_verified' => 1, 'roles' => ['faculty']]);
        $user = UserService::create($userData);

        $faculty = Faculty::where('user_id', $user->id)->firstOrFail();
        $faculty = $faculty->fill($request->only(['school_id', 'meta', 'description']));
        $faculty->save();

        return new FacultyAdminResource($faculty);
    }

    public function update(Faculty $faculty, Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:smartstudy_schools,id',
            'meta' => 'sometimes|json|nullable',
            'description' => 'sometimes'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        $faculty->fill($request->only(['school_id', 'meta', 'description']));
        $faculty->save();

        return new FacultyAdminResource($faculty);
    }

    public function destroy(Faculty $faculty)
    {
        Gate::authorize('delete');

        $faculty->delete();

        $faculty->user->delete();

        return response()->json([], 204);
    }
}
