<?php

namespace Vtlabs\Smartstudy\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Vtlabs\Smartstudy\Models\Quiz;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Smartstudy\Models\Student;
use Vtlabs\Smartstudy\Models\Quizoption;
use Vtlabs\Smartstudy\Filters\QuizFilter;
use Vtlabs\Smartstudy\Models\Quizattempt;
use Vtlabs\Smartstudy\Models\Quizquestion;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Smartstudy\Models\Quizleaderboard;
use Illuminate\Validation\ValidationException;
use Vtlabs\Smartstudy\Http\Resources\QuizResource;
use Vtlabs\Smartstudy\Http\Resources\QuizattemptResource;
use Vtlabs\Smartstudy\Http\Resources\QuizleaderboardResource;
use Vtlabs\Smartstudy\Http\Resources\QuizattemptSummaryResource;

class QuizController extends Controller
{
    public function __construct()
    {
        if (array_key_exists('HTTP_AUTHORIZATION', $_SERVER)) {
            $this->middleware('auth:api');
        }
    }
    
    public function index(Request $request)
    {
        $request->validate([
            'school' => 'sometimes|exists:smartstudy_schools,id',
            'schoolclass' => 'sometimes|exists:smartstudy_school_classes,id',
            'subject' => 'sometimes|exists:smartstudy_subjects,id'
        ]);

        $quiz = Quiz::filter($request->all(), QuizFilter::class)->orderBy('quiz_date', 'desc');

        return QuizResource::collection($quiz->paginate());
    }

    public function attempt(Quiz $quiz, Request $request)
    {
        $request->validate([
            'questions' => 'required|array',
            'questions.*.question_id' => 'required',
            'questions.*.option_id' => 'required',
            'questions.*.skipped' => 'required|boolean',
        ]);

        $student = Student::where('user_id', Auth::id())->firstOrFail();

        // check if already attempted
        if (Quizattempt::attempted($student->id, $quiz->id)->exists()) {
            throw ValidationException::withMessages(['quiz' => 'Already attempted']);
        }

        $totalCorrect = 0;
        $totalQuestions = $quiz->questions()->count();

        foreach ($request->questions as $questionData) {
            $question = Quizquestion::find($questionData['question_id']);
            $option = Quizoption::find($questionData['option_id']);
            $isCorrect = !$questionData['skipped'] ? $option->is_correct : false;
            Quizattempt::create([
                'quiz_id' => $quiz->id,
                'student_id' => $student->id,
                'question_id' => $question->id,
                'option_id' => $option->id,
                'is_correct' => $isCorrect,
                'is_skipped' => $questionData['skipped'] ?? false
            ]);

            if ($isCorrect) {
                $totalCorrect++;
            }
        }

        // update leaderboard
        $percentage = ($totalCorrect / $totalQuestions) * 100;
        Quizleaderboard::create([
            'score' => $percentage,
            'quiz_id' => $quiz->id,
            'student_id' => $student->id,
            'class_id' => $quiz->class_id,
            'subject_id' => $quiz->subject_id,
        ]);

        // quiz attempt summary
        return new QuizattemptSummaryResource([
            'quiz_id' => $quiz->id,
            'student_id' => $student->id,
            'score' => $percentage,
            'leaderboard_rank' => 0
        ]);
    }

    public function leaderboard(Quiz $quiz, Request $request)
    {
        $leaderboard = DB::select('SELECT *, FIND_IN_SET( score, (SELECT GROUP_CONCAT( DISTINCT score ORDER BY score DESC ) FROM smartstudy_quiz_leaderboards) ) AS leaderboard_rank FROM smartstudy_quiz_leaderboards');
        
        return QuizleaderboardResource::collection($leaderboard);
    }

    public function insights()
    {
        $student = Student::findByUser(Auth::id());
        
        // $perfomanceChartData = DB::table('smartstudy_quiz_leaderboards as l')->select(DB::raw('DATE(o.created_at) as created_at'), DB::raw('count(*) as total'))
        // ->whereDate('o.created_at', '>', Carbon::now()->subDays($days))
        // ->whereDate('o.created_at', '<=', Carbon::now())
        // ->join('statuses as s', 'o.id', '=', 's.model_id')
        // ->where('s.model_type', '=', Order::class)
        // ->groupBy(DB::raw($groupBy));
    }
}
