<?php

namespace Vtlabs\Smartstudy\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Smartstudy\Models\Quiz;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Smartstudy\Models\Quizoption;
use Vtlabs\Smartstudy\Filters\QuizFilter;
use Vtlabs\Smartstudy\Models\Quizquestion;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Smartstudy\Http\Resources\Admin\QuizAdminResource;

class QuizController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'school' => 'sometimes|exists:smartstudy_schools,id'
        ]);

        $quizs = Quiz::filter($request->all(), QuizFilter::class);

        if ($request->pagination == '0') {
            $quizs = $quizs->get();
        } else {
            $quizs = $quizs->paginate();
        }

        return QuizAdminResource::collection($quizs);
    }

    public function show(Quiz $quiz)
    {
        return new QuizAdminResource($quiz);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'subject_id' => 'required|exists:smartstudy_subjects,id',
            'quiz_date' => 'required',
            'duration_minutes' => 'required|numeric',
            'meta' => 'sometimes|json|nullable',
            'questions' => 'required|array',
            'questions.*.question' => 'required',
            'questions.*.options' => 'required|array',
            'questions.*.options.*.title' => 'required',
            'questions.*.options.*.is_correct' => 'required',
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null,
            "class_id" => 1
        ]);

        $quiz = Quiz::create($request->only(['title', 'description', 'school_id', 'class_id', 'subject_id', 'quiz_date', 'duration_minutes', 'meta']));

        // handle quiz questions
        foreach ($request->questions as $questionData) {
            $question = Quizquestion::create([
                'quiz_id' => $quiz->id,
                'question' => $questionData['question']
            ]);

            foreach ($questionData['options'] as $option) {
                Quizoption::create([
                    'question_id' => $question->id,
                    'title' => $option['title'],
                    'is_correct' => $option['is_correct']
                ]);
            }
        }

        return new QuizAdminResource($quiz);
    }

    public function update(Quiz $quiz, Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'school_id' => 'required|exists:smartstudy_schools,id',
            'class_id' => 'required|exists:smartstudy_school_classes,id',
            'subject_id' => 'required|exists:smartstudy_subjects,id',
            'quiz_date' => 'required',
            'duration_minutes' => 'required|numeric',
            'meta' => 'sometimes|json|nullable',
            'questions' => 'required|array',
            'questions.*.question' => 'required',
            'questions.*.options' => 'required|array',
            'questions.*.options.*.title' => 'required',
            'questions.*.options.*.is_correct' => 'required',
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null,
            "class_id" => 1
        ]);

        $quiz->fill($request->only(['title', 'description', 'school_id', 'class_id', 'subject_id', 'quiz_date', 'duration_minutes', 'meta']));
        $quiz->save();

        // handle quiz questions
        $quiz->questions()->delete();
        foreach ($request->questions as $questionData) {
            $question = Quizquestion::create([
                'quiz_id' => $quiz->id,
                'question' => $questionData['question']
            ]);

            foreach ($questionData['options'] as $option) {
                Quizoption::create([
                    'question_id' => $question->id,
                    'title' => $option['title'],
                    'is_correct' => $option['is_correct']
                ]);
            }
        }

        return new QuizAdminResource($quiz);
    }

    public function destroy(Quiz $quiz)
    {
        Gate::authorize('delete');

        $quiz->delete();

        return response()->json([], 204);
    }
}
