<?php

namespace Vtlabs\Ride\Models;

use Carbon\Carbon;
use EloquentFilter\Filterable;
use Vtlabs\Core\Models\Setting;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Rennokki\Plans\Traits\HasPlans;
use Vtlabs\Ride\Models\VehicleType;
use Rennokki\Rating\Traits\CanBeRated;
use Illuminate\Database\Eloquent\Model;
use Rennokki\Rating\Contracts\Rateable;
use Vtlabs\Core\Traits\CoreHasMediaTrait;
use Spatie\MediaLibrary\HasMedia\HasMedia;

class Driver extends Model implements Rateable, HasMedia
{
    use Filterable,
        CoreHasMediaTrait,
        HasPlans,
        CanBeRated;

    protected $table = 'ride_drivers';

    protected $guarded = [];

    protected $casts = [
        'meta' => 'json',
        'is_verified' => 'integer',
        'is_online' => 'integer',
        'is_riding' => 'integer',
        'current_latitude' => 'float',
        'current_longitude' => 'float',
        'distance_remaining' => 'integer'
    ];

    public static function findByUser($userId)
    {
        return Driver::where('user_id', $userId)->firstOrFail();
    }

    public static function search($user, $latitude, $longitude, $vehicleTypeId = null, $excludeDriverIds = [])
    {
        $distanceSetting = Setting::where('key', 'distance_limit')->first();
        $rideCompleteDistance = Setting::where('key', 'ride_complete_distance_meters')->first()->value;

        $distanceDelta = (int)$distanceSetting->value;

        $acceptTimeSetting = Setting::where('key', 'ride_accept_minutes_limit')->first();
        $rideAcceptMinutesDelta = (int)$acceptTimeSetting->value;

        $reassignRideTimeSetting = Setting::where('key', 'reassign_ride_minutes_limit')->first();
        $reassignRideMinutesDelta = (int)$reassignRideTimeSetting->value;

        // show drivers present with in particular distance
        $subqueryDistance = "ST_Distance_Sphere(Point(current_longitude,"
            . " current_latitude),"
            . " Point($longitude, $latitude ))"
            . " as distance";

        $subqueryDistanceWhere = "ST_Distance_Sphere(Point(current_longitude,"
            . " current_latitude),"
            . " Point($longitude, $latitude ))"
            . " < " . $distanceDelta;

        $drivers = Driver::select(
            '*',
            DB::raw($subqueryDistance)
        )
            ->whereRaw($subqueryDistanceWhere)
            ->where('is_online', 1)
            ->where('distance_remaining', '<=',  $rideCompleteDistance)
            //->where('is_riding', 0)
            ->whereRaw("(current_latitude != 0.0 and  current_longitude != 0.0)");

        if ($vehicleTypeId) {
            $drivers = $drivers->where('vehicle_type_id', $vehicleTypeId);
        }

        // filter out drivers who recently rejected or got cancelled
        $driverIds = Ride::select('driver_id')->where('user_id', $user->id)
            ->whereDate('created_at', '>=', Carbon::now()->subMinutes($reassignRideMinutesDelta)->format("Y-m-d H:i:s"))
            ->where(function ($query) {
                $query->where('status', 'rejected')
                    ->orWhere('status', 'pending')
                    ->orWhere('status', 'cancelled');
            })->get()->pluck('driver_id')->toArray();

        $driverIds = array_merge($driverIds, $excludeDriverIds);
        $drivers = $drivers->whereNotIn('id', $driverIds);

        // driver must be online
        $drivers = $drivers->where('is_online', true);

        // driver must not be busy in other ride
        // $drivers = $drivers->where('is_riding', false);

        // driver's profile must be verified
        $drivers = $drivers->where('is_verified', true);

        $drivers->orderBy('distance')->get();

        return $drivers;
    }

    public function releaseDriver()
    {
        $this->distance_remaining = 0;
        $this->is_riding = 0;
        $this->save();
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function rides()
    {
        return $this->hasMany('App\Models\Ride');
    }

    public function vehicleType()
    {
        return $this->belongsTo(VehicleType::class);
    }
}
