<?php

namespace Vtlabs\Ride\Listeners;

use Vtlabs\Core\Models\User\User;
use Vtlabs\Ride\Events\UpdateRide;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Mail;


class UpdateRideListener
{
    private $event;
    private $ride;
    private $notification;

    /**
     * Create the event listener.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Handle the event.
     *
     * @param  UpdateRide $event
     * @return void
     */
    public function handle(UpdateRide $event)
    {
        try {
            $this->event = $event;
            $this->ride = $event->ride;
            $this->notification = $event->notification;

            if ($this->ride->status == "cancelled") {
                $this->ride->driver->releaseDriver($this->ride->driver);
                $this->ride->driver->user->sendPushNotification(
                    'driver',
                    __('vtlabs_ride::messages.notification_ride_cancelled_driver_title', ['id' => $this->ride->id]),
                    __('vtlabs_ride::messages.notification_ride_cancelled_driver_body')
                );
            } else if ($this->ride->status == "rejected") {
                $this->ride->driver->releaseDriver($this->ride->driver);

                // send notification to user
                // while rejecting expired rides we do not want to send notification, that's why 
                // first check if notification is needs to be sent or not
                if ($this->notification) {
                    $this->ride->user->sendPushNotification(
                        'customer',
                        __('vtlabs_ride::messages.notification_ride_rejected_user_title', ['id' => $this->ride->id]),
                        __('vtlabs_ride::messages.notification_ride_rejected_user_body')
                    );
                }
            } else if ($this->ride->status == "accepted") {

                // send notification to user
                $this->ride->user->sendPushNotification(
                    'customer',
                    __('vtlabs_ride::messages.notification_ride_accepted_user_title', ['id' => $this->ride->id]),
                    __('vtlabs_ride::messages.notification_ride_accepted_user_body')
                );
            } else if ($this->ride->status == "ongoing") {
                
                // send notification to user
                $this->ride->user->sendPushNotification(
                    'customer',
                    __('vtlabs_ride::messages.notification_ride_ongoing_user_title', ['id' => $this->ride->id]),
                    __('vtlabs_ride::messages.notification_ride_ongoing_user_body')
                );
            } else if ($this->ride->status == "complete") {
                $this->ride->driver->releaseDriver($this->ride->driver);

                $finalFare = $this->ride->final_fare;

                $settings = CoreHelper::settingsAsDictionary();

                // calculate admin's share

                $adminFeeSetting = $settings['admin_fee_in_percent'] ?? 0;
                $adminEarning = ceil($finalFare * (int) $adminFeeSetting) / 100;

                // calculate driver earnings
                $driverEarning = $this->ride->final_fare - $adminEarning;

                // credit driver's wallet
                $this->ride->driver->user->deposit(
                    $driverEarning,
                    'deposit',
                    [
                        'description' => 'Ride #' . $this->ride->id,
                        'type' => 'earnings',
                        'source_title' => 'Earning',
                        'source' => 'ride',
                        'source_id' => $this->ride->id,
                        'source_amount' => $this->ride->total,
                        'source_payment_type' => $this->ride->payment->paymentMethod->type == 'postpaid' ? 'COD' : 'Online'
                    ]
                );

                if ($this->ride->paymentMethod->slug == 'COD') {
                    // in case of cash payment, withdraw admin earning from driver's wallet
                    $this->ride->driver->user->forceWithdraw(
                        $adminEarning,
                        'withdraw',
                        [
                            'description' => 'Cash collected',
                            'type' => 'cash_collected',
                            'source' => 'ride',
                            'source_id' => $this->ride->id,
                            'source_amount' => $adminEarning,
                            'source_payment_type' => 'COD'
                        ]
                    );
                }

                // if admin has earned something from this ride, create a transaction for admin's earning
                if ($adminEarning !== 0) {
                    User::find(1)->deposit($adminEarning, 'deposit', [
                        'description' => 'Ride #' . $this->ride->id,
                        'type' => 'earnings',
                        'source' => 'ride',
                        'source_id' => $this->ride->id,
                        'source_amount' => $this->ride->final_fare,
                        'source_payment_type' => $this->ride->payment->paymentMethod->type == 'postpaid' ? 'COD' : 'Online'
                    ]);
                }

                // if ($this->order->user->email) {
                //     Mail::to($this->order->user)->send(new RideInvoice($this->ride));
                // }
            }
        } catch (\Exception $ex) {
            Log::error('Exception: Notification not sent', [$ex->getMessage(), $ex->getTraceAsString()]);
        }
    }
}
