<?php

namespace Vtlabs\Ride\Jobs;

use Vtlabs\Ride\Models\Ride;
use Illuminate\Bus\Queueable;
use Vtlabs\Ride\Models\Driver;
use Vtlabs\Ride\Events\NewRide;
use Vtlabs\Ride\Events\UpdateRide;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Ride\Filters\DriverFilter;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class FindNextDriver implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $ride;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(Ride $ride)
    {
        $this->ride = $ride;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        // If a request goes to one driver and did not accept it in 15 seconds app must go try another driver nearest. 
        if ($this->ride->status == 'pending') {
            $newDriver = Driver::filter([
                'lat' => $this->ride->latitude_from,
                'long' => $this->ride->longitude_from,
                'vehicleType' => $this->ride->vehicle_type_id,
                'riding' => 0,
                'exclude_past_drivers_for_ride' => $this->ride->id
            ], DriverFilter::class)->first();

            if ($newDriver) {
                $this->ride->driver->releaseDriver();

                $this->ride->driver_id = $newDriver->id;
                $this->ride->save();
                event(new NewRide($this->ride->refresh()));

                $settings = CoreHelper::settingsAsDictionary();
                $acceptDuration = $settings['ride_accept_duration'] ?? 0;
                FindNextDriver::dispatch($this->ride)->delay(now()->addSeconds($acceptDuration));
            } else {
                // cancel the ride
                $this->ride->setStatus('cancelled');
                $this->ride->save();

                event(new UpdateRide($this->ride->refresh()));
            }
        }
    }
}
