<?php

namespace Vtlabs\Ride\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Ride\Models\Ride;
use Vtlabs\Ride\Models\Driver;
use Vtlabs\Ride\Events\NewRide;
use Vtlabs\Ride\Events\UpdateRide;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Ride\Filters\RideFilter;
use Vtlabs\Ride\Models\VehicleType;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Ride\Jobs\FindNextDriver;
use Vtlabs\Ride\Filters\DriverFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Ride\Http\Resources\RideResource;
use Vtlabs\Core\Services\GoogleDistanceService;

class RideController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'status' => 'sometimes',
            'upcoming' => 'sometimes'
        ]);

        $rides = Ride::filter($request->all(), RideFilter::class);

        $rides = $rides->orderBy('ride_on', 'desc');

        return RideResource::collection($rides->paginate());
    }

    public function store(Request $request)
    {
        $request->validate([
            'type' => 'required|in:ride,intercity,courier',
            'vehicle_type_id' => 'required|exists:ride_vehicle_types,id',
            'ride_on' => 'required|date_format:Y-m-d H:i',
            'address_from' => 'required|string',
            'latitude_from' => 'required|numeric',
            'longitude_from' => 'required|numeric',
            'address_to' => 'required|string',
            'latitude_to' => 'required|numeric',
            'longitude_to' => 'required|numeric',
            'is_scheduled' => 'sometimes|numeric'
        ]);

        $settings = CoreHelper::settingsAsDictionary();
        $user = Auth::user();


        // calculate estimated distance and time, we are doing this before anything else so that in case
        // Google Api Key is not enabled or billing not enabled we do not create ride            
        $googleDistanceService = new GoogleDistanceService($request->latitude_from, $request->longitude_from, $request->latitude_to, $request->longitude_to);
        [$distance, $time] = $googleDistanceService->distanceAndTime();

        // calculate and update estimated fares
        $vehicleType = VehicleType::find($request->vehicle_type_id);
        $estimatedFare = $vehicleType->calculateFare($distance, $time);

        // search for driver
        $driver = Driver::filter([
            'lat' => $request->latitude_from,
            'long' => $request->longitude_from,
            'vehicleType' => $request->vehicle_type_id,
            'riding' => 0
        ], DriverFilter::class)->firstOrFail();
        $driver->is_riding = 1;
        $driver->distance_remaining = $distance * 1000; // km to meters
        $driver->save();

        $ride = new Ride();
        $ride->fill($request->all());
        $ride->user_id = Auth::user()->id;
        $ride->driver_id = $driver->id;

        // update estimated fares
        $ride->estimated_fare = $estimatedFare;
        $ride->estimated_distance = $distance;
        $ride->estimated_time = $time;

        // calculate estimated distance and time to pickup        
        $googleDistanceService = new GoogleDistanceService($driver->current_latitude, $driver->current_longitude, $request->latitude_from, $request->longitude_from);
        [$distance, $time] = $googleDistanceService->distanceAndTime();

        $ride->estimated_pickup_distance = $distance;
        $ride->estimated_pickup_time = $time;

        $ride->save();

        // payment
        $paymentMethodSlug = $request->payment_method_slug;
        $payment = Auth::user()->createPayment($ride, $estimatedFare, $paymentMethodSlug);

        event(new NewRide($ride));

        // find next driver if current driver do not accept ride in given time
        $acceptDuration = $settings['ride_accept_duration'] ?? 0;
        FindNextDriver::dispatch($ride)->delay(now()->addSeconds($acceptDuration));

        $ride->refresh();

        return new RideResource($ride);
    }

    public function update(Ride $ride, Request $request)
    {
        $request->validate([
            'status' => 'required'
        ]);
        if (Auth::user()->provider->id != $ride->driver_id) {
            return response()->json(['message' => 'Cannot update ride'], 400);
        }

        $old_status = $ride->status;

        if ($request->status == 'rejected') {
            // find next driver if current driver reject ride
            FindNextDriver::dispatch($ride);
        } else {

            $ride->setStatus($request->status);
            $ride->save();

            if ($ride->status == 'ongoing') {
                $ride->ride_start_at = Carbon::now();
            }

            if ($ride->status == 'complete') {
                // recalculate and update final fare

                $googleDistanceService = new GoogleDistanceService($ride->latitude_from, $ride->longitude_from, $ride->driver->current_latitude, $ride->driver->current_longitude);
                [$distance, $time] = $googleDistanceService->distanceAndTime();

                // calculate time
                $rideTime = Carbon::now()->diffInMinutes($ride->ride_start_at);

                $vehicleType = VehicleType::find($ride->vehicle_type_id);
                $finalFare = $vehicleType->calculateFare($distance, $time);

                $ride->final_fare = $finalFare;
                $ride->final_distance = $distance;
                $ride->final_time = $rideTime;
                $ride->ride_ends_at = Carbon::now();
            }

            $ride->save();
        }

        return new RideResource($ride);

        return response()->json($ride->refresh());
    }

    public function cancel(Ride $ride)
    {
        $ride->setStatus('cancelled');
        $ride->save();

        event(new UpdateRide($ride));

        return new RideResource($ride);
    }

    public function fareEstimate(Request $request)
    {
        $request->validate([
            'vehicle_type_id' => 'required',
            'latitude_from' => 'required|numeric',
            'longitude_from' => 'required|numeric',
            'latitude_to' => 'required|numeric',
            'longitude_to' => 'required|numeric',
        ]);

        try {
            $googleDistanceService = new GoogleDistanceService(
                $request->latitude_from,
                $request->longitude_from,
                $request->latitude_to,
                $request->longitude_to
            );
            [$distance, $time] = $googleDistanceService->distanceAndTime();
        } catch (\Exception $ex) {
            return response()->json(['message' => $ex->getMessage()], 400);
        }

        $vehicleType = VehicleType::find($request->vehicle_type_id);
        $estimatedFare = $vehicleType->calculateFare($distance, $time);

        return response()->json([
            'fare' => $estimatedFare,
            'distance' => $distance,
            'time' => $time
        ]);
    }
}
