<?php

namespace Vtlabs\Ride\Http\Controllers\Api;

use Illuminate\Http\Request;
use Vtlabs\Ride\Models\Driver;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Ride\Filters\DriverFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Ride\Http\Resources\DriverResource;

class DriverController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'vehicle_type_id' => 'sometimes|exists:vehicle_types,id',
            'lat' => 'sometimes|numeric',
            'long' => 'sometimes|numeric'
        ]);

        request()->merge([
            'is_online' => 1
        ]);

        $drivers = Driver::filter($request->all(), DriverFilter::class);

        $drivers = $drivers->orderBy('distance')->limit(10)->get();

        return DriverResource::collection($drivers);
    }

    public function show(Driver $driver)
    {
        return new DriverResource($driver);
    }

    public function showMyProfile(Request $request)
    {
        $driver = Driver::findByUser(Auth::user()->id);
        return new DriverResource($driver);
    }

    public function update(Request $request)
    {
        $request->validate([
            'vehicle_type_id' => 'sometimes|exists:ride_vehicle_types,id',
            'is_online' => 'sometimes|boolean',
            'current_latitude' => 'sometimes|numeric',
            'current_longitude' => 'sometimes|numeric',
            'distance_remaining' => 'sometimes|numeric',
            'meta' => 'sometimes|json',
        ]);

        request()->merge(['meta' => $request->meta ? json_decode($request->meta, true) : null]);

        $driver = Driver::findByUser(Auth::user()->id);
        $driver->fill($request->only(['vehicle_type_id', 'is_online', 'current_latitude', 'current_longitude', 'distance_remaining', 'meta']));
        $driver->save();

        return new DriverResource($driver);
    }

    public function ratingStore(Driver $driver, Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric',
            'review' => 'required'
        ]);

        $user = Auth::user();

        // $user->unrate($vendor);
        $user->rate($driver, $request->rating, $request->review);

        // update average rating of vendor for stats
        $driver->average_ratings +=  $driver->averageRating(User::class);
        $driver->save();

        return response()->json([], 200);
    }

    public function ratingSummary(Driver $driver)
    {
        return response()->json([
            "average_rating" => $driver->averageRating(User::class),
            "total_ratings" => $driver->raters(User::class)->count(),
            "summary" => DB::table('ratings')->selectRaw('count(*) as total, ROUND(rating) as rounded_rating')
                ->where('rateable_type', Driver::class)
                ->where('rateable_id', $driver->id)
                ->where('rater_type', User::class)
                ->groupBy('rounded_rating')
                ->get()
        ]);
    }
}
