<?php

namespace Vtlabs\Ride\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Ride\Models\Driver;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Ride\Http\Resources\Admin\DriverAdminResource;

class DriverController extends Controller
{
    public function index(Request $request)
    {
        return DriverAdminResource::collection(Driver::latest()->paginate());
    }

    public function show(Driver $driver)
    {
        return new DriverAdminResource($driver);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'meta' => 'sometimes|json|nullable',
            'is_verified' => 'required|boolean'
        ]);

        request()->merge([
            'meta' => $request->meta ? json_decode($request->meta, true) : null
        ]);

        // create user for driver first  
        $userData = collect($request->only(['email', 'mobile_number', 'password', 'name']));
        $userData = $userData->merge(['mobile_verified' => 1, 'roles' => ['driver']]);
        $user = UserService::create($userData);

        $driver = new Driver();
        $driver->user_id = $user->id;

        $driver->fill($request->only([
            'meta', 'is_verified'
        ]));
        $driver->save();

        return new DriverAdminResource($driver->refresh());
    }

    public function update(Driver $driver, Request $request)
    {
        $request->validate([
            'meta' => 'sometimes|json|nullable',
            'is_verified' => 'required|boolean'
        ]);

        request()->merge([
            'meta' => $request->meta ? json_decode($request->meta, true) : null
        ]);

        $driver->fill($request->only([
            'meta', 'is_verified'
        ]));
        $driver->save();

        return new DriverAdminResource($driver->refresh());
    }


    public function destroy(Driver $driver, Request $request)
    {
        Gate::authorize('delete');

        $driver->delete();

        return response()->json([], 204);
    }
}
