<?php

namespace Vtlabs\Provider\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Contracts\Auth\Access\Gate;
use Vtlabs\Appointment\Models\Appointment;
use Vtlabs\Provider\Filters\ProviderFilter;
use Vtlabs\Provider\Models\ProviderProfile;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Provider\Models\ProviderAvailability;
use Vtlabs\Provider\Http\Resources\ProviderRatingResource;
use Vtlabs\Provider\Http\Resources\ProviderProfileResource;

class ProviderProfileController extends Controller
{
    public function __construct(Gate $gate)
    {
        if (array_key_exists('HTTP_AUTHORIZATION', $_SERVER)) {
            $this->middleware('auth:api');
        }
    }

    public function index(Request $request)
    {
        $request->validate([
            'search' => 'sometimes',
            'category' => 'sometimes|exists:categories,id',
            'lat' => 'sometimes|numeric',
            'long' => 'sometimes|numeric'
        ]);

        $providers = ProviderProfile::filter($request->all(), ProviderFilter::class)->latest();

        return ProviderProfileResource::collection($providers->paginate());
    }

    public function show(ProviderProfile $profile, Request $request)
    {
        return new ProviderProfileResource($profile);
    }

    public function showMyProfile(Request $request)
    {

        $providerProfile = ProviderProfile::findByUser(Auth::user()->id);
        return new ProviderProfileResource($providerProfile);
    }

    public function showProfileByUser(User $user, Request $request)
    {
        $providerProfile = ProviderProfile::findByUser($user->id);
        return new ProviderProfileResource($providerProfile);
    }

    public function update(Request $request)
    {
        $request->validate([
            'name' => 'sometimes|string',
            'details' => 'sometimes|string',
            'address' => 'sometimes',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
            'categories' => 'sometimes|array',
            'categories.*.id' => 'sometimes|exists:categories,id',
            'categories.*.fee' => 'sometimes|numeric',
            'subcategories' => 'sometimes|array',
            'subcategories.*.id' => 'sometimes|exists:categories,id',
            'subcategories.*.fee' => 'sometimes|numeric',
            'image_urls' => 'sometimes|array',
            'meta' => 'sometimes|json',
            'fee' => 'sometimes|numeric',
            'availability' => 'sometimes|array',
            'availability.*.days' => 'required|in:sun,mon,tue,wed,thu,fri,sat',
            'availability.*.from' => 'required|date_format:H:i',
            'availability.*.to' => 'required|date_format:H:i',
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta, true) : null
        ]);

        $providerProfile = ProviderProfile::findByUser(Auth::user()->id);

        $providerProfile->fill($request->only(['address', 'longitude', 'latitude', 'meta', 'fee']));

        CoreHelper::fillTranslation($providerProfile, $request->only(['name', 'details']));

        $providerProfile->save();

        // attach categories
        if ($request->categories) {
            $selectedCategories = [];
            foreach ($request->categories as $category) {
                $selectedCategories[$category['id']] = ['fee' => isset($category['fee']) ? $category['fee'] : 0.0];
            }
            $providerProfile->categories()->sync($selectedCategories);
        }

        // attach subcategories
        if ($request->categories) {
            $selectedSubCategories = [];
            foreach ($request->subcategories as $subcategory) {
                $selectedSubCategories[$subcategory['id']] = ['fee' => isset($subcategory['fee']) ? $subcategory['fee'] : 0.0];
            }
            $providerProfile->subcategories()->sync($selectedSubCategories);
        }

        // images
        if (!empty($request->image_urls)) {
            $newMediaItems = [];
            foreach ($request->image_urls as $imageUrl) {
                $newMediaItems[] = $providerProfile->addMediaFromUrl($imageUrl)->toMediaCollection("images");
            }
            $providerProfile->clearMediaCollectionExcept('images', $newMediaItems);
        }

        // attach availablities
        if ($request->availability) {
            $providerProfile->availabilities()->delete();
            foreach ($request->availability as $availability) {
                ProviderAvailability::create([
                    'provider_profile_id' => $providerProfile->id,
                    'days' => $availability['days'],
                    'from' => $availability['from'],
                    'to' => $availability['to']
                ]);
            }
        }

        return new ProviderProfileResource($providerProfile);
    }

    public function ratingList(ProviderProfile $providerProfile, Request $request)
    {
        return ProviderRatingResource::collection($providerProfile->raters(User::class)->orderByDesc('pivot_created_at')->paginate());
    }

    public function ratingStore(ProviderProfile $providerProfile, Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric',
            'review' => 'required',
            'meta' => 'sometimes|json'
        ]);

        $request->merge([
            'meta' => $request->meta ? json_decode($request->meta, true) : null
        ]);

        $user = Auth::user();

        $user->rate($providerProfile, $request->rating, $request->review, $request->meta);

        return response()->json([], 200);
    }

    public function ratingSummary(ProviderProfile $providerProfile)
    {
        return response()->json([
            "average_rating" => $providerProfile->averageRating(User::class),
            "total_ratings" => $providerProfile->raters(User::class)->count(),
            "summary" => DB::table('ratings')->selectRaw('count(*) as total, ROUND(rating) as rounded_rating')
                ->where('rateable_type', ProviderProfile::class)
                ->where('rateable_id', $providerProfile->id)
                ->where('rater_type', User::class)
                ->groupBy('rounded_rating')
                ->get()
        ]);
    }

    public function favourites(Request $request)
    {
        return ProviderProfileResource::collection(Auth::user()->favorite(ProviderProfile::class)->flatten());
    }

    public function toggleFavourite(ProviderProfile $providerProfile)
    {
        $providerProfile->toggleFavorite();

        return response()->json([]);
    }

    public function schedule(ProviderProfile $providerProfile, Request $request)
    {
        $request->validate([
            'days' => 'required|numeric',
            'start_from' => 'required|date_format:Y-m-d'
        ]);

        $settings = CoreHelper::settingsAsDictionary();

        if (isset($settings['schedule_interval_in_minutes'])) {
            $schedule = [];
            $interval = $settings['schedule_interval_in_minutes'];
            // @TODO: availabilities in ProviderProfile
            $dayWiseAvailability = $this->mapAvailabilityToDay($providerProfile->availabilities);

            $daysPeriod = Carbon::parse($request->start_from)->daysUntil($request->days);

            foreach ($daysPeriod as $date) {
                if (!isset($dayWiseAvailability[strtolower($date->shortEnglishDayOfWeek)])) {
                    continue;
                }

                $timeFrom = $dayWiseAvailability[strtolower($date->shortEnglishDayOfWeek)]['from'];
                $timeTo = $dayWiseAvailability[strtolower($date->shortEnglishDayOfWeek)]['to'];

                if ($timeFrom && $timeTo) {
                    $timePeriod = Carbon::parse($date->format('Y-m-d') . ' ' . $timeFrom)->minutesUntil($date->format('Y-m-d') . ' ' . $timeTo, $interval);
                    foreach ($timePeriod as $time) {
                        $isScheduled = Appointment::where('date', $time->format('Y-m-d'))->where('time_from', $time->format('H:i'))->exists();
                        $schedule[] = ['time' => $time->format('Y-m-d H:i'), 'is_scheduled' => $isScheduled];
                    }
                }
            }
            return response()->json(['schedule' => $schedule]);
        }

        return response()->json(['message' => 'Setting not set'], 400);
    }

    private function mapAvailabilityToDay($availabilities)
    {
        $availabilityMap = [];
        foreach ($availabilities as $availability) {
            $availabilityMap[$availability['days']] = ['from' => $availability['from'], 'to' => $availability['to']];
        }

        return $availabilityMap;
    }
}
