<?php

namespace Vtlabs\Provider\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Provider\Models\Referral;
use Rennokki\Plans\Models\PlanUsageLog;
use Vtlabs\Provider\Models\ReferralLog;
use Vtlabs\Appointment\Models\Appointment;
use Vtlabs\Provider\Events\NewAppointment;
use Vtlabs\Provider\Models\ProviderProfile;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Provider\Events\UpdateAppointment;
use Vtlabs\Provider\Filters\AppointmentFilter;
use Vtlabs\Appointment\Services\AppointmentService;
use Vtlabs\Provider\Http\Resources\AppointmentResource;

class AppointmentController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'appointer' => 'required_without:appointee|exists:users,id',
            'appointee' => 'required_without:appointer|exists:provider_profiles,id'
        ]);

        $appointments = Appointment::filter($request->all(), AppointmentFilter::class);

        if ($request->past) {
            $appointments = $appointments->orderBy('updated_at', 'desc');
        } else {
            $appointments = $appointments->orderBy('date', 'desc')->orderBy('time_from', 'desc');
        }

        return AppointmentResource::collection($appointments->paginate());
    }

    public function store(ProviderProfile $providerProfile, Request $request)
    {
        $user = Auth::user();

        $request->validate(AppointmentService::validationRules());

        $appointment = $user->appoint($providerProfile, $request->all());

        // handle provider referral code
        if ($request->referral_code) {
            $referral = Referral::where('referral_code', $request->referral_code)->first();

            if ($referral->user_id != $user->id && !ReferralLog::where('user_id', Auth::id())->where('referral_id', $referral->id)->exists()) {
                ReferralLog::create([
                    'user_id' => Auth::id(),
                    'referral_id' => $referral->id
                ]);

                // update the referral usage count
                $referral->usage_counter += 1;
                $referral->save();
            }
        }

        // handle free service logic for referrals
        $referral = Referral::where('user_id', Auth::id())->where('provider_profile_id', $providerProfile->id)->first();
        $settings = CoreHelper::settingsAsDictionary();
        if($referral && $referral->usage_counter / $settings['provider_referral_count'] >= 1) {
            $meta = !empty($appointment->meta) ? $appointment->meta : null;
            if($meta) {
                $appointment->meta = [];
            }
            $meta['discounted_service'] = 1;
            $appointment->meta = $meta;
            $appointment->save();

            $referral->usage_counter -= $settings['provider_referral_count'];
            $referral->save();
        }

        event(new NewAppointment($appointment));

        return new AppointmentResource($appointment);
    }

    public function update(Appointment $appointment, Request $request)
    {
        $request->validate([
            'status' => ['required', Rule::in(config('vtlabs_appointment.appointment.status_list'))]
        ]);

        $old_status = $appointment->status;

        if ($request->status == 'accepted') {
            // can accept the job?
            $canAccept = false;
            $provider = ProviderProfile::findByUser(Auth::id());
            $leadsRemainingForToday = null;

            if ($provider->hasActiveSubscription()) {
                $subscription = $provider->activeSubscription();
                $leadsUsedToday = PlanUsageLog::getTodayUsage($subscription->id);
                $limit = $subscription->features()->code('leads_per_day')->first()->limit / 30;
                $leadsRemainingForToday = $limit - $leadsUsedToday;

                if ($leadsRemainingForToday > 0) {
                    $canAccept = true;
                }
            }

            if (!$canAccept) {
                return response()->json(["message" => 'Not enough credits left'], 403);
            }
        }

        $appointment->touch();

        AppointmentService::updateStatus($appointment, $request->status);

        if ($old_status != $appointment->status && $appointment->status == 'accepted') {
            // deduct the credits
            $provider = ProviderProfile::findByUser(Auth::id());
            $subscription = $provider->activeSubscription();
            $subscription->consumeFeature('leads_per_day', 1);
            PlanUsageLog::create(["subscription_id" => $subscription->id]);
        }

        event(new UpdateAppointment($appointment));

        return new AppointmentResource($appointment->fresh());
    }
}
