<?php

namespace Vtlabs\Provider\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Provider\Models\Hospital;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Provider\Models\Availability;
use Vtlabs\Provider\Filters\ProviderFilter;
use Vtlabs\Provider\Models\ProviderProfile;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Provider\Models\ProviderAvailability;
use Vtlabs\Provider\Http\Resources\Admin\ProviderProfileAdminResource;
use Rennokki\Plans\Models\PlanModel;

class ProviderProfileController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'name' => 'sometimes'
        ]);

        $providers = ProviderProfile::filter($request->all(), ProviderFilter::class)->latest();

        return ProviderProfileAdminResource::collection($providers->paginate());
    }

    public function show(ProviderProfile $providerprofile, Request $request)
    {
        return new ProviderProfileAdminResource($providerprofile);
    }

    public function store(Request $request)
    {
        $request->validate([
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'name_translations' => 'required|json|translation',
            'details_translations' => 'required|json|translation',
            'is_verified' => 'required|boolean',
            'address' => 'sometimes',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
            'fee' => 'sometimes|numeric',
            'categories' => 'required|array|exists:categories,id',
            'subcategories' => 'required|array|exists:categories,id',
            'availability' => 'sometimes|array',
            'availability.*.days' => 'required|in:sun,mon,tue,wed,thu,fri,sat',
            'availability.*.from' => 'required',
            'availability.*.to' => 'required',
            'images.*' => 'sometimes|image',
            'meta' => 'sometimes|json|nullable',
            'plan_id' => 'sometimes|exists:plans,id'
        ]);

        request()->merge([
            "name" => request()->input('_name'),
            "details" => request()->input('_details') ?? null
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta, true) : null
        ]);

        // create user for provider first  
        $userName = $request->name[array_keys(($request->name))[0]];
        $userData = collect($request->only(['email', 'mobile_number', 'password']));
        $userData = $userData->merge(['name' => $userName, 'mobile_verified' => 1, 'roles' => ['provider']]);
        $user = UserService::create($userData);

        $providerProfile = ProviderProfile::where('user_id', $user->id)->firstOrFail();
        $providerProfile = $providerProfile->fill($request->only([
            'name', 'details', 'address', 'longitude', 'latitude', 'is_verified', 'meta', 'fee'
        ]));
        $providerProfile->save();

        // attach categories
        $providerProfile->categories()->sync($request->categories);

        // attach subcategories
        $providerProfile->subcategories()->sync($request->subcategories);

        // attach availablities
        if ($request->availability) {
            foreach ($request->availability as $availability) {
                ProviderAvailability::create([
                    'provider_profile_id' => $providerProfile->id,
                    'days' => $availability['days'],
                    'from' => $availability['from'],
                    'to' => $availability['to']
                ]);
            }
        }

        // images
        if (!empty($request->images)) {
            foreach ($request->images as $image) {
                $providerProfile->addMedia($image)->toMediaCollection("images");
            }
        }

        // plan assignment
        if ($request->plan_id) {
            $plan = PlanModel::find($request->plan_id);
            if ($providerProfile->hasActiveSubscription()) {
                $providerProfile->cancelCurrentSubscription();
            }
            $providerProfile->subscribeTo($plan, $plan->duration);
            $providerProfile->plan_sort_order = $plan->sort_order;
        }

        return new ProviderProfileAdminResource($providerProfile);
    }

    public function update(ProviderProfile $providerprofile, Request $request)
    {
        $request->validate([
            'name_translations' => 'required|json|translation',
            'details_translations' => 'required|json|translation',
            'is_verified' => 'required|boolean',
            'address' => 'sometimes',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
            'fee' => 'sometimes|numeric',
            'categories' => 'required|array|exists:categories,id',
            'subcategories' => 'required|array|exists:categories,id',
            'availability' => 'sometimes|array',
            'availability.*.days' => 'required|in:sun,mon,tue,wed,thu,fri,sat',
            'availability.*.from' => 'required',
            'availability.*.to' => 'required',
            'images.*' => 'sometimes|image',
            'meta' => 'sometimes|json|nullable',
            'plan_id' => 'sometimes|exists:plans,id'
        ]);

        request()->merge([
            "name" => request()->input('_name'),
            "details" => request()->input('_details') ?? null
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta, true) : null
        ]);

        $providerprofile->fill($request->only([
            'name', 'details', 'address', 'longitude', 'latitude', 'is_verified', 'meta', 'fee'
        ]));

        $providerprofile->save();

        // attach categories
        $providerprofile->categories()->sync($request->categories);

        // attach subcategories
        $providerprofile->subcategories()->sync($request->subcategories);

        // attach availablities
        if ($request->availability) {
            $providerprofile->availabilities()->delete();
            foreach ($request->availability as $availability) {
                ProviderAvailability::create([
                    'provider_profile_id' => $providerprofile->id,
                    'days' => $availability['days'],
                    'from' => $availability['from'],
                    'to' => $availability['to']
                ]);
            }
        }

        // images
        if (!empty($request->images)) {
            $providerprofile->clearMediaCollection('images');
            foreach ($request->images as $image) {
                $providerprofile->addMedia($image)->toMediaCollection("images");
            }
        }

        if ($request->profile_image) {
            $providerprofile->user->clearMediaCollection('images');
            $providerprofile->user->addMedia($request->profile_image)->toMediaCollection("images");
        }

        // plan assignment
        if ($request->plan_id) {
            $plan = PlanModel::find($request->plan_id);
            if ($providerprofile->hasActiveSubscription()) {
                $providerprofile->cancelCurrentSubscription();
            }
            $providerprofile->subscribeTo($plan, $plan->duration);
            $providerprofile->plan_sort_order = $plan->sort_order;
        }

        return new ProviderProfileAdminResource($providerprofile);
    }

    public function destroy(ProviderProfile $providerprofile)
    {
        Gate::authorize('delete');

        $providerprofile->delete();

        return response()->json([], 200);
    }
}
