<?php

namespace Vtlabs\Provider\Http\Controllers\Api\Admin;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Vtlabs\Appointment\Models\Appointment;
use Vtlabs\Ecommerce\Models\Order;
use Vtlabs\Ecommerce\Filters\OrderFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Ecommerce\Http\Resources\Admin\OrderAdminResource;

class DashboardController extends Controller
{
    public function appointmentAnalytics(Request $request)
    {
        $request->validate([
            'period' => 'required|in:week,month,year,all'
        ]);

        $days = 7;
        $period = $request->period;
        $groupBy = "DATE('o.created_at')";
        $label = 'DATE(o.created_at) as created_at';

        switch ($period) {
            case 'week':
                $days = 7;
                $groupBy = 'DATE(o.created_at)';
                $label = 'DATE(o.created_at) as created_at';
                break;
            case 'month':
                $days = 30;
                $groupBy = 'DATE(o.created_at)';
                $label = 'DATE(o.created_at) as created_at';
                break;
            case 'year':
                $days = 365;
                $groupBy = 'MONTH(o.created_at)';
                $label = 'MONTH(o.created_at) as created_at';
                break;
            case 'all':
                $days = 365 * 5;
                $groupBy = 'YEAR(o.created_at)';
                $label = 'YEAR(o.created_at) as created_at';
                break;
        }

        $appointmentsChartData = DB::table('appointments as a')->select(DB::raw($label), DB::raw('count(*) as total'))
            ->whereDate('a.created_at', '>', Carbon::now()->subDays($days))
            ->whereDate('a.created_at', '<=', Carbon::now())
            ->join('statuses as s', 'a.id', '=', 's.model_id')
            ->where('s.model_type', '=', Appointment::class)
            ->groupBy(DB::raw($groupBy));

        $completeAppointmentsChartData = clone $appointmentsChartData;
        $completeAppointmentsChartData = $completeAppointmentsChartData->where('s.name', 'complete')->get()->mapWithKeys(function ($item) {
            return [$item->created_at => $item->total];
        })->all();

        $otherAppointmentsChartData = clone $appointmentsChartData;
        $otherAppointmentsChartData = $otherAppointmentsChartData->where('s.name', '<>', 'complete')->get()->mapWithKeys(function ($item) {
            return [$item->created_at => $item->total];
        })->all();

        $appointmentsChartData = $appointmentsChartData->get();

        $summary = [
            ["title" => "Total", "value" => Appointment::whereRaw("1=1")->count()],
            ["title" => "Last Month", "value" => Appointment::whereDate('created_at', '>', Carbon::now()->subDays(30))->count()],
            ["title" => "Last Week", "value" => Appointment::whereDate('created_at', '>', Carbon::now()->subDays(7))->count()],
            ["title" => "Today", "value" => Appointment::whereDate('created_at', '=', Carbon::now())->count()]
        ];

        $chartLabel = array_map(function ($date) use ($period) {
            if ($period == 'week') {
                $in = date_create($date);
                return date_format($in, "D");
            }
            if ($period == 'month') {
                $in = date_create($date);
                return date_format($in, "d");
            }
            if ($period == 'year') {
                $in = date_create('2013-10-27'); // create a sample date object
                $in->setDate($in->format('Y'), $date, $in->format('d'));
                return $in->format("M");
            }
            if ($period == 'all') {
                $in = date_create('2013-10-27'); // create a sample date object
                $in->setDate($date, $in->format('m'), $in->format('d'));
                return $in->format("Y");
            }
            return $date;
        }, $appointmentsChartData->pluck('created_at')->toArray());

        $allAppointmentsCount = [];
        $completeAppointmentsCount = [];
        $otherAppointmentsCount = [];

        foreach ($appointmentsChartData as $appointmentData) {
            $allAppointmentsCount[]  = $appointmentData->total;
            $completeAppointmentsCount[] = $completeAppointmentsChartData[$appointmentData->created_at] ?? 0;
            $otherAppointmentsCount[] = $otherAppointmentsChartData[$appointmentData->created_at] ?? 0;
        }

        return response()->json([
            "chart" => [
                "chartLabel" => $chartLabel,
                "linesData" => [
                    $allAppointmentsCount,
                    $otherAppointmentsCount,
                    $completeAppointmentsCount,
                ]
            ],
            "summary" => $summary
        ]);
    }
}
