<?php

namespace Vtlabs\Payment\Http\Controllers\Api;

use Razorpay\Api\Api;
use Yabacon\Paystack;
use PayFast\PayFastPayment;
use Illuminate\Http\Request;
use Vtlabs\Core\Models\User\User;
use Vtlabs\Payment\Models\Payment;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Payment\Models\PaymentMethod;
use Vtlabs\Core\Services\FirebaseService;
use Vtlabs\Payment\Events\OnWalletPayment;
use Vtlabs\Core\Http\Controllers\Controller;
use net\authorize\api\contract\v1 as AnetAPI;
use Vtlabs\Payment\Filters\PaymentMethodFilter;
use net\authorize\api\controller as AnetController;
use Vtlabs\Payment\Http\Resources\PaymentMethodResource;
use Cartalyst\Stripe\Laravel\Facades\Stripe;

class PaymentController extends Controller
{
    public function paymentMethods(Request $request)
    {
        $paymentMethods = PaymentMethod::filter(array_merge(["enabled" => true], $request->all()), PaymentMethodFilter::class)->get();

        return PaymentMethodResource::collection($paymentMethods);
    }

    public function failedPayment(Payment $payment, Request $request)
    {
        $payment->setStatus('failed');
        $payment->save();

        return response(["success" => false, "message" => 'Payment failed'], 400);
    }

    public function makeStripePayment(Payment $payment, Request $request)
    {
        $amount = number_format((float)$payment->amount, 2, '.', '');

        $settings = CoreHelper::settingsAsDictionary();

        $currency = isset($settings['currency_code']) ? strtolower($settings['currency_code']) : 'usd';

        $stripe = Stripe::make(env('STRIPE_SECRET'));

        $paymentIntent = $stripe->paymentIntents()->create([
            'amount' => $amount,
            'currency' => strtolower($currency),
            'description' => 'Payment',
            'payment_method_types' => [
                'card',
            ]
        ]);

        return view('vendor.payments.stripe.index', [
            'stripeKey' => env('STRIPE_KEY'),
            'paymentIntentClientSecret' => $paymentIntent['client_secret'],
            'returnUrl' => route('apipayment.stripe.callback', ['payment' => $payment->id, 'amount' => $amount])
        ]);
    }

    public function callbackStripePayment(Payment $payment, Request $request)
    {
        $request->validate([
            'payment_intent' => 'required',
            'payment_intent_client_secret' => 'required',
            'redirect_status' => 'required'
        ]);
        $stripe = Stripe::make(env('STRIPE_SECRET'));
        $paymentIntent = $stripe->paymentIntents()->find($request->payment_intent);

        if ($paymentIntent['status'] == 'succeeded') {
            $payment->setStatus('paid');
            $payment->save();
            header('Location: ' . route('apipayment.stripe.status', ['payment' => $payment->id, 'result' => 'success']));
            exit;
        }

        $payment->setStatus('failed');
        $payment->save();
        header('Location: ' . route('apipayment.stripe.status', ['payment' => $payment->id, 'result' => 'error']));
        exit;
    }

    public function statusStripePayment(Request $request)
    {
        echo $request->result;
    }

    // public function makeStripePayment(Payment $payment, Request $request)
    // {
    //     $request->validate([
    //         'token' => 'required'
    //     ]);

    //     $settings = CoreHelper::settingsAsDictionary();
    //     $currency = isset($settings['currency_code']) ? strtolower($settings['currency_code']) : 'usd';

    //     $parameters = [
    //         'amount' => $payment->amount,
    //         'currency' => $currency,
    //         'token' => $request->token
    //     ];

    //     $stripeGateway = new StripeGateway();

    //     try {
    //         $stripeResponse = $stripeGateway->purchase($payment, $parameters);
    //         if ($stripeResponse->isSuccessfull()) {
    //             return response(["success" => true, "message" => "Payment succesfull"]);
    //         }
    //     } catch (\Exception $ex) {
    //         $payment->setStatus('failed');
    //         $payment->save();

    //         return response()->json(["success" => false, "message" => $ex->getMessage()], 400);
    //     }
    // }

    public function makePayuPayment(Payment $payment, Request $request)
    {
        $request->validate([
            'result' => 'required|in:success,failed'
        ]);

        if ($request->result == 'success') {
            $payment->setStatus('paid');
            $payment->save();

            return response(["success" => true, "message" => "Payment succesfull"]);
        }

        if ($request->result == 'failed') {
            $payment->setStatus('failed');
            $payment->save();

            return response(["success" => false, "message" => "Payment failed"]);
        }
    }

    public function makePaystackPayment(Payment $payment, Request $request)
    {
        $paystack = new Paystack(env('PAYSTACK_SECURE_KEY'));
        try {
            $tranx = $paystack->transaction->initialize([
                'amount' => $payment->amount * 100,
                'email' => $payment->payer->email,         // unique to customers
                'reference' => (string)time(), // unique to transactions,
                'callback_url' => url('/api/payment/paystack/callback/' . $payment->id . '?amount=' . $payment->amount)
            ]);
        } catch (\Yabacon\Paystack\Exception\ApiException $e) {
            die($e->getMessage());
        }

        // store transaction reference so we can query in case user never comes back
        // perhaps due to network issue
        // save_last_transaction_reference($tranx->data->reference);

        // redirect to page so User can pay
        header('Location: ' . $tranx->data->authorization_url);
    }


    public function paystackCallback(Payment $payment, Request $request)
    {
        $reference = isset($_GET['reference']) ? $_GET['reference'] : '';
        if (!$reference) {
            header('Location: ' . url('/api/payment/paystack/status/' . $payment->id . '?result=error'));
        }

        // initiate the Library's Paystack Object
        $paystack = new Paystack(env('PAYSTACK_SECURE_KEY'));
        try {
            // verify using the library
            $tranx = $paystack->transaction->verify([
                'reference' => $reference, // unique to transactions
            ]);
        } catch (\Exception $e) {
            $payment->setStatus('failed');
            $payment->save();
            header('Location: ' . url('/api/payment/paystack/status/' . $payment->id . '?result=error'));
            exit;
        }

        if ($tranx && 'success' === $tranx->data->status) {
            $payment->setStatus('paid');
            $payment->save();
            header('Location: ' . url('/api/payment/paystack/status/' . $payment->id . '?result=success'));
            exit;
        } else {
            $payment->setStatus('failed');
            $payment->save();
            header('Location: ' . url('/api/payment/paystack/status/' . $payment->id . '?result=error'));
            exit;
        }
    }

    public function paystackStatus(Request $request)
    {
        echo $request->result;
    }

    public function makeWalletPayment(Payment $payment, Request $request)
    {
        $user = User::find($payment->payer_id);

        if ($user->canWithdraw($payment->amount)) {
            $payment->setStatus('paid');
            $payment->save();

            // withdraw from wallet is responsibility of listerner because we want to attach the payble_type object in
            // transaction meta
            event(new OnWalletPayment($payment->fresh()));

            return response(["success" => true, "message" => "Payment succesfull"]);
        }

        $payment->setStatus('failed');
        $payment->save();

        return response(["success" => false, "message" => "Insufficient balance in wallet"], 400);
    }

    public function makeAuthorizeNetPayment(Payment $payment, Request $request)
    {
        $request->validate([
            'name' => 'required',
            'card_number' => 'required',
            'exp_month' => 'required',
            'exp_year' => 'required',
            'cvv' => 'required'
        ]);

        $input = $request->input();
        $errorMessage = 'There were some issue with the payment. Please try again later.';

        /* Create a merchantAuthenticationType object with authentication details
          retrieved from the constants file */
        $merchantAuthentication = new AnetAPI\MerchantAuthenticationType();
        $merchantAuthentication->setName(env('ANET_MERCHANT_LOGIN_ID'));
        $merchantAuthentication->setTransactionKey(env('ANET_MERCHANT_TRANSACTION_KEY'));

        // Set the transaction's refId
        $refId = 'ref-' . $payment->id . '-' . time();
        $cardNumber = preg_replace('/\s+/', '', $input['card_number']);

        // Create the payment data for a credit card
        $creditCard = new AnetAPI\CreditCardType();
        $creditCard->setCardNumber($cardNumber);
        $creditCard->setExpirationDate($input['exp_year'] . "-" . $input['exp_month']);
        $creditCard->setCardCode($input['cvv']);

        // Add the payment data to a paymentType object
        $paymentOne = new AnetAPI\PaymentType();
        $paymentOne->setCreditCard($creditCard);

        // Create a TransactionRequestType object and add the previous objects to it
        $transactionRequestType = new AnetAPI\TransactionRequestType();
        $transactionRequestType->setTransactionType("authCaptureTransaction");
        $transactionRequestType->setAmount($payment->amount);
        $transactionRequestType->setPayment($paymentOne);

        // Assemble the complete transaction request
        $requests = new AnetAPI\CreateTransactionRequest();
        $requests->setMerchantAuthentication($merchantAuthentication);
        $requests->setRefId($refId);
        $requests->setTransactionRequest($transactionRequestType);

        // Create the controller and get the response
        $controller = new AnetController\CreateTransactionController($requests);
        $response = $controller->executeWithApiResponse(\net\authorize\api\constants\ANetEnvironment::SANDBOX);

        if ($response != null) {
            // Check to see if the API request was successfully received and acted upon
            if ($response->getMessages()->getResultCode() == "Ok") {
                // Since the API request was successful, look for a transaction response
                // and parse it to display the results of authorizing the card
                $tresponse = $response->getTransactionResponse();

                if ($tresponse != null && $tresponse->getMessages() != null) {
                    $payment->setStatus('paid');
                    $payment->save();

                    return response(["success" => true, "message" => "Payment succesfull"]);
                } else {
                    if ($tresponse->getErrors() != null) {
                        $errorMessage = $tresponse->getErrors()[0]->getErrorText();
                    }
                }
            } else {
                $tresponse = $response->getTransactionResponse();

                if ($tresponse != null && $tresponse->getErrors() != null) {
                    $errorMessage = $tresponse->getErrors()[0]->getErrorText();
                } else {
                    $errorMessage = $response->getMessages()->getMessage()[0]->getText();
                }
            }
        } else {
            $errorMessage = "No response returned";
        }

        $payment->setStatus('failed');
        $payment->save();

        return response(["success" => false, "message" => $errorMessage]);
    }

    // bambora payment gateway
    public function makeBamboraPayment(Payment $payment, Request $request)
    {
        $request->validate([
            'name' => 'required',
            'card_number' => 'required',
            'exp_month' => 'required',
            'exp_year' => 'required',
            'cvv' => 'required'
        ]);

        $merchant_id = env('BAMBORA_MERCHANT_ID'); //INSERT MERCHANT ID (must be a 9 digit string)
        $api_key = env('BAMBORA_API_KEY'); //INSERT API ACCESS PASSCODE
        $api_version = 'v1'; //default
        $platform = 'api'; //default

        //Create Beanstream Gateway
        $beanstream = new \Beanstream\Gateway($merchant_id, $api_key, $platform, $api_version);

        //Example Card Payment Data
        $payment_data = array(
            'order_number' => $payment->id,
            'amount' => $payment->amount,
            'payment_method' => 'card',
            'card' => array(
                'name' => $request->name,
                'number' => $request->card_number,
                'expiry_month' => $request->card_number,
                'expiry_year' => $request->card_number,
                'cvd' => $request->cvv,
            )
        );
        $complete = TRUE; //set to FALSE for PA

        //Try to submit a Card Payment
        try {

            $result = $beanstream->payments()->makeCardPayment($payment_data, $complete);

            /*
            * Handle successful transaction, payment method returns
            * transaction details as result, so $result contains that data
            * in the form of associative array.
            */
            $payment->setStatus('paid');
            $payment->save();

            return response(["success" => true, "message" => "Payment succesfull"]);
        } catch (\Beanstream\Exception $e) {
            /*
            * Handle transaction error, $e->code can be checked for a
            * specific error, e.g. 211 corresponds to transaction being
            * DECLINED, 314 - to missing or invalid payment information
            * etc.
            */

            $payment->setStatus('failed');
            $payment->save();

            return response(["success" => false, "message" => $e->getMessage()]);
        }
    }

    // telebirr payment gateway


    private function telebirrEncryptRSA($data, $public)
    {
        $pubPem = chunk_split($public, 64, "\n");
        $pubPem = "-----BEGIN PUBLIC KEY-----\n" . $pubPem . "-----END PUBLIC KEY-----\n";
        $public_key = openssl_pkey_get_public($pubPem);
        if (!$public_key) {
            die('invalid public key');
        }
        $crypto = '';
        foreach (str_split($data, 117) as $chunk) {
            $return = openssl_public_encrypt($chunk, $cryptoItem, $public_key);
            if (!$return) {
                return ('fail');
            }
            $crypto .= $cryptoItem;
        }
        $ussd = base64_encode($crypto);
        return $ussd;
    }

    public function makeTelebirrPayment(Payment $payment, Request $request)
    {
        $appKey = env('TELEBIRR_APP_KEY');
        $appId = env('TELEBIRR_APP_ID');

        $data = [
            'outTradeNo' => $payment->id . '-' . uniqid("", true),
            'subject' => 'Payment for pay id #' . $payment->id,
            'totalAmount' => $payment->amount,
            'shortCode' => env('TELEBIRR_SHORT_CODE'),
            'notifyUrl' => url('/api/payment/telebirr/callback/' . $payment->id),
            'returnUrl' => url('/api/payment/telebirr/callback/' . $payment->id),
            'receiveName' => env('TELEBIRR_RECEIVER_NAME'),
            'appId' => $appId,
            'timeoutExpress' => '30',
            'nonce' => uniqid("", true),
            'timestamp' => strval(time() * 1000)
        ];

        $data['appKey'] = $appKey;

        ksort($data);
        $StringA = '';
        foreach ($data as $k => $v) {
            if ($StringA == '') {
                $StringA = $k . '=' . $v;
            } else {
                $StringA = $StringA . '&' . $k . '=' . $v;
            }
        }

        $StringB = hash("sha256", $StringA);


        $sign = strtoupper($StringB);

        $ussdjson = json_encode($data);

        $publicKey = env('TELEBIRR_PUBLIC_KEY');
        $ussd = $this->telebirrEncryptRSA($ussdjson, $publicKey);
        $requestMessage = [
            'appid' => $appId,
            'sign' => $sign,
            'ussd' => $ussd
        ];

        $api = 'http://196.188.120.3:10443/service-openup/toTradeWebPay';

        // Create a new cURL resource
        $ch = curl_init($api);

        $payload = json_encode($requestMessage);

        // Attach encoded JSON string to the POST fields
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);

        // Set the content type to application/json
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));

        // Return response instead of outputting
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        // Execute the POST request
        $result = curl_exec($ch);

        // sample response 
        // {"code":200,"data":{"toPayUrl":"http://196.188.120.3:11443/ammwebpay/#/?transactionNo=202112071640201468213792884588545"},"message":"Operation successful","dateTime":1638884420976,"path":null,"errorDetails":{},"extraData":{}}

        if (curl_errno($ch)) {
            return response(["message" => curl_error($ch)], 400);
        } else {
            // check the HTTP status code of the request
            $resultStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            if ($resultStatus != 200) {
                return response(["message" => 'Request failed: HTTP status code: ' . $resultStatus], 400);
            }
            $response = json_decode($result, true);

            if ($response['code'] !== 200) {
                return response(["message" => $response['message']], 400);
            }
            header('location:' . $response['data']['toPayUrl']);
        }

        // Close cURL resource
        curl_close($ch);
    }

    public function telebirrCallback(Payment $payment, Request $request)
    {
        $responseCode = $request->code;
        $msg = $request->msg;

        if ($msg == 'success') {

            $payment->setStatus('paid');
            $payment->save();
            header('Location: ' . url('/api/payment/telebirr/status/' . $payment->id . '?result=success'));
        } else {

            $payment->setStatus('failed');
            $payment->save();
            header('Location: ' . url('/api/payment/telebirr/status/' . $payment->id . '?result=error'));
        }
    }

    public function telebirrStatus(Request $request)
    {
        echo $request->result;
    }

    public function esewaIndex($payment, Request $request)
    {
        $payment = Payment::findOrFail($payment);

        $scd = env('ESEWA_SCD', null);
        $payUrl = env('ESEWA_URL', null);

        if (!$scd || !$payUrl) {
            abort(400);
        }

        return view('vendor.payments.esewa', [
            'amount' => $payment->amount,
            'su' => url('/api/payment/esewa/' . $payment->id . '/callback'),
            'fu' => url('/api/payment/esewa/' . $payment->id . '/callback'),
            'pid' => $payment->id,
            'scd' => $scd,
            'payUrl' => $payUrl
        ]);
    }

    public function esewaCallback(Payment $payment, Request $request)
    {
        $request->validate([
            'refId' => 'sometimes'
        ]);

        if ($request->refId) {
            $payment->setStatus('paid');
            $payment->save();

            header('Location: ' . url('/api/payment/esewa/status/' . $payment->id . '?result=success'));
            exit;
        } else {
            $payment->setStatus('failed');
            $payment->save();

            header('Location: ' . url('/api/payment/esewa/status/' . $payment->id . '?result=error'));
            exit;
        }
    }

    public function esewaStatus(Request $request)
    {
        echo $request->result;
    }

    public function razorpayCreateOrder(Payment $payment, Request $request)
    {
        $request->validate([]);

        $settings = CoreHelper::settingsAsDictionary();
        $currency = isset($settings['currency_code']) ? strtoupper($settings['currency_code']) : 'usd';

        $api = new Api(env("RAZORPAY_API_KEY"), env("RAZORPAY_API_SECRET"));

        $orderData = [
            'receipt'         => 'rcptid_' . $payment->id . '_' . time(),
            'amount'          => $payment->amount * 100,
            'currency'        => $currency
        ];

        $razorpayOrder = $api->order->create($orderData);

        // save razorpay order id in payment meta
        $meta = $payment->meta;
        $meta['razorpay_order_id'] = $razorpayOrder->id;
        $payment->meta = $meta;
        $payment->save();

        return response()->json(['razorpay_order_id' => $razorpayOrder->id]);
    }

    public function razorpayVerifyOrder(Payment $payment, Request $request)
    {
        $request->validate([
            'razorpay_payment_id' => 'required',
            'razorpay_signature' => 'required'
        ]);

        $api = new Api(env("RAZORPAY_API_KEY"), env("RAZORPAY_API_SECRET"));

        $attributes  = array('razorpay_signature'  => $request->razorpay_signature,  'razorpay_payment_id'  => $request->razorpay_payment_id,  'razorpay_order_id' => $payment->meta['razorpay_order_id']);

        try {
            $api->utility->verifyPaymentSignature($attributes);
            $payment->setStatus('paid');
            $payment->save();

            return response(["success" => true, "message" => "Payment succesfull"]);
        } catch (\Exception $ex) {
            $payment->setStatus('failed');
            $payment->save();

            return response(["success" => false, "message" => $ex->getMessage()], 400);
        }
    }

    public function razorpayFailedPayment(Payment $payment, Request $request)
    {
        $payment->setStatus('failed');
        $payment->save();

        return response(["success" => false, "message" => 'Payment failed'], 400);
    }

    // Digipay payment gateway
    public function makeDigipayPayment(Payment $payment, Request $request)
    {
        $appUserId = env('DIGIPAY_USERID');
        $appUserPassword = env('DIGIPAY_USER_PASSWORD');
        $appHashkey = env('DIGIPAY_HASHKEY');
        $appMerchantCode = env('DIGIPAY_MERCHANT_CODE');

        $refId = $payment->id . '-' . time();

        $settings = CoreHelper::settingsAsDictionary();
        $currency = isset($settings['currency_code']) ? strtoupper($settings['currency_code']) : 'usd';

        $auth = base64_encode($appUserId . ":" . $appUserPassword);
        $signatureData = $appMerchantCode . $payment->amount . $refId . $currency;
        $signature = hash_hmac('sha256', $signatureData, $appHashkey, false);

        $data = [
            'merchant' => $appMerchantCode,
            'amount' => $payment->amount,
            'currency' => $currency,
            'billref' => $refId,
            'cbk_url' => url('/api/payment/digipay/callback/' . $payment->id),
            'web_cbk_url' => url('/api/payment/digipay/callback/' . $payment->id),
            'signature' => $signature
        ];

        $payload = json_encode($data);

        $api = 'https://gateway.digipay.africa/api' . '/v2/transaction/paylink';

        // Create a new cURL resource
        $ch = curl_init($api);

        // Attach encoded JSON string to the POST fields
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);

        // Set the content type to application/json
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json', 'Authorization: Basic ' . $auth));

        // Return response instead of outputting
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        // Execute the POST request
        $result = curl_exec($ch);

        if (curl_errno($ch)) {
            return response(["message" => curl_error($ch)], 400);
        } else {
            // check the HTTP status code of the request
            $resultStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            if ($resultStatus != 200) {
                return response(["message" => 'Request failed: HTTP status code: ' . $resultStatus], 400);
            }
            $response = json_decode($result, true);

            if ($response['code'] !== 200) {
                return response(["message" => $response['message']], 400);
            }
            header('location:' . $response['redirect_url']);
        }

        // Close cURL resource
        curl_close($ch);
    }

    public function digipayCallback(Payment $payment, Request $request)
    {
        $request->validate([
            'paystatus' => 'required'
        ]);

        if ($request->paystatus && $request->paystatus == 'success') {
            $payment->setStatus('paid');
            $payment->save();

            header('Location: ' . url('/api/payment/digipay/status/' . $payment->id . '?result=success'));
            exit;
        } else {
            $payment->setStatus('failed');
            $payment->save();

            header('Location: ' . url('/api/payment/digipay/status/' . $payment->id . '?result=error'));
            exit;
        }
    }

    public function digipayStatus(Request $request)
    {
        echo $request->result;
    }

    public function nexiIndex($payment, Request $request)
    {
        $payment = Payment::findOrFail($payment);
        $settings = CoreHelper::settingsAsDictionary();

        $payUrl = env('NEXI_URL');
        $mid = env('NEXI_MID');
        $secret = env('NEXI_SECRET');
        $version = "2";
        $orderid = time() . $payment->id;
        $orderDesc = "Payment for order";
        $orderAmount = $payment->amount;
        $currency = isset($settings['currency_code']) ? strtoupper($settings['currency_code']) : 'EUR';
        $billCountry = "GR";
        $billZip = "24100";
        $billCity = "xxxx";
        $billAddress = "xxxx";
        $confirmUrl = route('apipayment.nexi.callback', ['payment' => $payment->id, 'result' => 'success']);
        $cancelUrl = route('apipayment.nexi.callback', ['payment' => $payment->id, 'result' => 'error']);
        $data = $version . $mid . $orderid . $orderDesc . $orderAmount . $currency . $billCountry . $billZip . $billCity . $billAddress . $confirmUrl . $cancelUrl . $secret;
        $digest = base64_encode(hash("sha256", utf8_encode($data), true));


        if (!$payUrl || !$payUrl) {
            abort(400);
        }

        return view('vendor.payments.nexi', [
            'payUrl' => $payUrl,
            'mid' => $mid,
            'version' => $version,
            'orderid' => $orderid,
            'orderDesc' => $orderDesc,
            'orderAmount' => $orderAmount,
            'currency' => $currency,
            'billCountry' => $billCountry,
            'billZip' => $billZip,
            'billCity' => $billCity,
            'billAddress' => $billAddress,
            'confirmUrl' => $confirmUrl,
            'cancelUrl' => $cancelUrl,
            'data' => $data,
            'digest' => $digest
        ]);
    }

    public function nexiCallback(Payment $payment, Request $request)
    {
        $request->validate([
            'result' => 'sometimes'
        ]);

        if ($request->result == 'success') {
            $payment->setStatus('paid');
            $payment->save();

            header('Location: ' . route('apipayment.nexi.status', ['payment' => $payment->id, 'result' => 'success']));
            exit;
        } else {
            $payment->setStatus('failed');
            $payment->save();

            header('Location: ' . route('apipayment.nexi.status', ['payment' => $payment->id, 'result' => 'error']));
            exit;
        }
    }

    public function nexiStatus(Request $request)
    {
        echo $request->result;
    }

    public function payfastPaymentForm(Payment $payment, Request $request)
    {
        $amount = $payment->amount;
        $refId = 'ref-' . $payment->id . '-' . time();

        $payfast = new PayFastPayment(
            [
                'merchantId' => '10031862',
                'merchantKey' => 'tlex091tcou5o',
                'passPhrase' => '123456789012',
                'testMode' => true
            ]
        );

        $data = [
            // Merchant details
            'return_url' => route('apipayment.payfast.callback', ['payment' => $payment->id, 'result' => 'success']),
            'cancel_url' => route('apipayment.payfast.callback', ['payment' => $payment->id, 'result' => 'cancel']),
            // 'notify_url' => route('apipayment.payfast.callback', ['payment' => $payment->id]),
            // Buyer details
            'email_address' => 'test@test.com',
            // Transaction details
            'm_payment_id' => $refId, //Unique payment ID to pass through to notify_url
            'amount' => $amount,
            'item_name' => 'Paymen#' . $payment->id
        ];

        $htmlForm = $payfast->custom->createFormFields($data, ['value' => 'PLEASE PAY', 'class' => 'button-cta']);

        echo $htmlForm . "<script type='text/javascript'>document.querySelector('.button-cta').click();</script>";
    }

    public function payfastPaymentCallback(Payment $payment, Request $request)
    {
        Log::info(json_encode(request()));
        $request->validate([
            'result' => 'sometimes'
        ]);

        if ($request->result == 'success') {
            $payment->setStatus('paid');
            $payment->save();

            header('Location: ' . route('apipayment.payfast.status', ['payment' => $payment->id, 'result' => 'success']));
            exit;
        } else {
            $payment->setStatus('failed');
            $payment->save();

            header('Location: ' . route('apipayment.payfast.status', ['payment' => $payment->id, 'result' => 'error']));
            exit;
        }
    }

    public function payfastStatus(Request $request)
    {
        echo $request->result;
    }

    public function makeVivawalletPayment(Payment $payment, Request $request)
    {
        $auth = base64_encode(env('VIVAWALLET_MERCHANT_ID') . ":" . env('VIVAWALLET_API_Key'));

        // $accessToken = 'xxx';

        $postFields  = [
            'amount'              => $payment->amount * 100,
            'customerTrns'        => 'Payment for order #' . $payment->payable_id
        ];

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL            => 'https://www.vivapayments.com/api/orders',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING       => '',
            CURLOPT_MAXREDIRS      => 10,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST  => 'POST',
            CURLOPT_CONNECTTIMEOUT  => 0,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_POSTFIELDS     => json_encode($postFields),
            CURLOPT_HTTPHEADER     => array(
                // "Authorization: Bearer $accessToken",
                "Authorization: Basic $auth",
                'Content-Type: application/json',
                'Accept: application/json'
            ),
        ));

        $response = curl_exec($curl); // {"orderCode":4201736414972602}

        if (curl_errno($curl)) {
            $message = curl_error($curl);
            curl_close($curl);
            return response(["message" => $message], 400);
        } else {
            // check the HTTP status code of the request
            $resultStatus = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);
            if ($resultStatus != 200) {
                return response(["message" => 'Request failed: HTTP status code: ' . $resultStatus], 400);
            }
        }

        $response = json_decode($response);

        header('Location: ' . 'https://www.vivapayments.com/web/checkout?ref=' . $response->OrderCode);
        exit;
    }

    public function callbackVivawallet(Payment $payment, Request $request)
    {
        $request->validate([
            'tid' => 'required'
        ]);

        $auth = base64_encode(env('VIVAWALLET_MERCHANT_ID') . ":" . env('VIVAWALLET_API_Key'));

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL            => 'https://www.vivapayments.com/api/transactions/',
            $request->tid,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING       => '',
            CURLOPT_MAXREDIRS      => 10,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST  => 'GET',
            CURLOPT_CONNECTTIMEOUT  => 0,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER     => array(
                "Authorization: Basic $auth",
                'Content-Type: application/json',
                'Accept: application/json'
            ),
        ));

        $response = curl_exec($curl); // {"orderCode":4201736414972602}

        if (curl_errno($curl)) {
            $message = curl_error($curl);
            curl_close($curl);
            return response(["message" => $message], 400);
        } else {
            // check the HTTP status code of the request
            $resultStatus = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);
            if ($resultStatus != 200) {
                return response(["message" => 'Request failed: HTTP status code: ' . $resultStatus], 400);
            }
        }

        $response = json_decode($response);

        if ($response->Success) {
            $payment->setStatus('paid');
            $payment->save();
            return response(["message" => 'Success'], 200);
        }

        $payment->setStatus('failed');
        $payment->save();

        return response(["message" => $response->ErrorText], 400);
    }

    public function makeSafirimpesaPayment(Payment $payment, Request $request)
    {
        $request->validate([
            'mobile' => 'required'
        ]);

        $creds = json_decode($payment->paymentMethod->meta, true);
        $baseUrl = $creds['SAFIRIMPESA_ENV'] == 'prod' ? 'https://api.safaricom.co.ke' : 'https://sandbox.safaricom.co.ke';
        $accessToken = $this->getSafirimpesaAccessToken($baseUrl, $creds['SAFIRIMPESA_CONSUMER_KEY'], $creds['SAFIRIMPESA_CONSUMER_SECRET']);
        date_default_timezone_set('Africa/Nairobi');
        $callbackurl = route('apipayment.safirimpesa.callback', ['payment' => $payment->id]);
        $passkey = $creds['SAFIRIMPESA_PASSKEY'];
        $businessShortCode = $creds['SAFIRIMPESA_SHORTCODE'];
        $timestamp = date('YmdHis');
        $password = base64_encode($businessShortCode . $passkey . $timestamp);
        $phone = $request->mobile;
        $amount = ceil($payment->amount);

        $requestUrl = $baseUrl . '/mpesa/stkpush/v1/processrequest';
        $headers = ['Content-Type:application/json', 'Authorization:Bearer ' . $accessToken];
        $payload = [
            'BusinessShortCode' => $businessShortCode,
            'Password' => $password,
            'Timestamp' => $timestamp,
            'TransactionType' => 'CustomerPayBillOnline',
            'Amount' => $amount,
            'PartyA' => $phone,
            'PartyB' => $businessShortCode,
            'PhoneNumber' => $phone,
            'CallBackURL' => $callbackurl,
            "AccountReference" => env('APP_NAME') . ' Payment',
            "TransactionDesc" => env('APP_NAME') . ' Payment',
        ];

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $requestUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => $headers,
        ));

        $response = curl_exec($curl);

        if (curl_errno($curl)) {
            $message = curl_error($curl);
            return response(["message" => $message], 400);
        }

        $resultStatus = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        if ($resultStatus != 200) {
            return response(["message" => 'Request failed: HTTP status code: ' . $resultStatus], 400);
        }

        return $response;
    }

    public function safirimpesaCallback(Payment $payment, Request $request)
    {
        $callbackResponse = file_get_contents('php://input');
        $data = json_decode($callbackResponse);
        $resultCode = $data->Body->stkCallback->ResultCode;

        if ($resultCode == 0) {
            $payment->setStatus('paid');
            $payment->save();
            list($response, $statusCode) = [["message" => 'Success'], 200];
        } else {
            $payment->setStatus('failed');
            $payment->save();
            list($response, $statusCode) = [["message" => $data->Body->stkCallback->ResultDesc], 200];
        }

        $this->updatePaymentOnFirebase($payment);

        return response(["message" => $response], $statusCode);
    }

    private function getSafirimpesaAccessToken($baseUrl, $consumerKey, $consumerSecret)
    {
        $auth = base64_encode($consumerKey . ":" . $consumerSecret);

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $baseUrl . '/oauth/v1/generate?grant_type=client_credentials',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_HTTPHEADER => array(
                'Authorization: Basic ' . $auth
            ),
        ));

        $response = curl_exec($curl);

        if (curl_errno($curl)) {
            return response(["message" => curl_error($curl)], 400);
        } else {
            $resultStatus = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            if ($resultStatus != 200) {
                return response(["message" => 'Request failed: HTTP status code: ' . $resultStatus], 400);
            }
        }

        $response = json_decode($response);

        return $response->access_token;
    }

    private function updatePaymentOnFirebase(Payment $payment)
    {
        try {
            $firebaseDatabase = FirebaseService::getDatabaseInstance();
            if ($firebaseDatabase) {
                $firebaseDatabase->getReference('/fire_app/payments/' . $payment->id . "/data")->set($payment);
            }
        } catch (\Exception $ex) {
            Log::error($ex);
        }
    }

    public function makePaiementproPayment(Payment $payment, Request $request)
    {
        $slug = $payment->paymentMethod->slug;
        $channel = implode('-', array_slice(explode('-', $slug), 1));
        ini_set("soap.wsdl_cache_enabled", 0);
        $url = "https://www.paiementpro.net/webservice/OnlineServicePayment_v2.php?wsdl";
        $client = new \SoapClient($url, array('cache_wsdl' => WSDL_CACHE_NONE));
        $array = array(
            'merchantId' => 'PP-F105',
            'countryCurrencyCode' => '952',
            'amount' => $payment->amount,
            'channel' => strtoupper($channel),
            'customerEmail' => 't@t.ci',
            'customerFirstName' => 'Thierry',
            'customerLastname' => 'Narcisse',
            'customerPhoneNumber' => '22507517917',
            'referenceNumber' => $payment->id . '-' . time(),
            'notificationURL' => route('apipayment.paiementpro.callback', ['payment' => $payment->id]),
            'returnURL' => route('apipayment.paiementpro.callback', ['payment' => $payment->id]),
            'description' => 'Payment for Payment #' . $payment->id

            // 'merchantId' => 'PP-F105',
            // 'countryCurrencyCode' => '952',
            // 'amount' => 1000,
            // 'channel' => 'CARD',
            // 'customerEmail' => 't@t.ci',
            // 'customerFirstName' => 'Test',
            // 'customerLastname' => 'name',
            // 'customerPhoneNumber' => '22507517917',
            // 'referenceNumber' => $payment . '-' . time(),
            // 'notificationURL' => route('apipayment.paiementpro.callback', ['payment' => $payment->id]),
            // 'returnURL' => route('apipayment.paiementpro.callback', ['payment' => $payment->id]),
            // 'description' => 'Payment for Payment #' . $payment->id
        );
        try {
            $response = $client->initTransact($array);

            if ($response->Code == 0) {
                header("Location:https://www.paiementpro.net/webservice/onlinepayment/processing_v2.php?sessionid=" . $response->Sessionid);
            }
        } catch (\Exception $e) {
            echo $e->getMessage();
        }
    }

    public function paiementproCallback(Payment $payment, Request $request)
    {
        Log::info($request->all());
        $result = 'failed';
        if ($request->responsecode == '0') {
            $result = 'success';
            $payment->setStatus('paid');
            $payment->save();

            header('Location: ' . route('apipayment.paiementpro.status', ['payment' => $payment->id, 'result' => 'success']));
            exit;
        } else {
            $result = 'failed';
            $payment->setStatus('failed');
            $payment->save();

            header('Location: ' . route('apipayment.paiementpro.status', ['payment' => $payment->id, 'result' => 'failed']));
            exit;
        }
    }

    public function paiementproStatus(Request $request)
    {
        echo $request->result;
    }
}
