<?php

namespace Vtlabs\Media\Models;

use EloquentFilter\Filterable;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Rennokki\Rating\Traits\CanBeRated;
use Illuminate\Database\Eloquent\Model;
use Rennokki\Rating\Contracts\Rateable;
use Vtlabs\Report\Traits\CanBeReported;
use Spatie\Translatable\HasTranslations;
use Overtrue\LaravelLike\Traits\Likeable;
use Vtlabs\Core\Traits\CoreHasMediaTrait;
use Spatie\MediaLibrary\HasMedia;
use BeyondCode\Comments\Traits\HasComments;
use ChristianKuri\LaravelFavorite\Traits\Favoriteable;
use Spatie\ModelStatus\HasStatuses;
use Vtlabs\Media\Traits\WithoutAppends;

class Media extends Model implements HasMedia, Rateable
{
    use Filterable,
        HasTranslations,
        Favoriteable,
        CoreHasMediaTrait,
        CanBeRated,
        Likeable,
        HasComments,
        CanBeReported,
        HasStatuses,
        WithoutAppends;

    public const STATUS_OPEN = 'open';
    public const STATUS_REVIEWED = 'reviewed';
    public const STATUS_ASSIGNED = 'assigned';
    public const STATUS_STARTED = 'started';
    public const STATUS_COMPLETE = 'complete';
    public const STATUS_OVERDUE = 'overdue';

    protected $table = 'media';

    protected $guarded = [];

    public $with = ['category', 'subcategories', 'content', 'media', 'statuses'];

    protected $casts = [
        'title' => 'array',
        'description' => 'array',
        'short_description' => 'array',
        'meta' => 'array',
        'price' => 'float',
        'is_public' => 'integer'
    ];

    protected $appends = ['favoritesCount', 'isFavourite'];

    public $availableMediaConversions = [];

    public $translatable = ['title', 'description', 'short_description'];

    public static function boot()
    {
        parent::boot();

        // on create
        static::created(function ($order) {
            $order->setStatus(config('vtlabs_media.media.status_default'));
        });
    }

    public static function saveHashtags($string) {
        preg_match_all('/#(\w+)/',$string, $matches);
        foreach($matches[0] as $match) {
            DB::table('media_hashtags')->updateOrInsert(['hashtag' => $match], []);
            DB::table('media_hashtags')->where('hashtag', $match)->increment('frequency');
        }
    }

    public static function deleteHashtags($string) {
        preg_match_all('/#(\w+)/',$string, $matches);
        foreach($matches[0] as $match) {
            $frequncency = DB::table('media_hashtags')->where('hashtag', $match)->pluck('frequency')->first();
            if($frequncency > 0) {
                DB::table('media_hashtags')->where('hashtag', $match)->decrement('frequency');
            }
        }
    }

    public function getIsFavouriteAttribute()
    {
        return $this->isFavorited();
    }

    public function category()
    {
        return $this->belongsTo(config('vtlabs_category.models.category'), 'category_id');
    }

    public function subcategories()
    {
        return $this->belongsToMany(config('vtlabs_category.models.category'), 'media_categories');
    }

    public function content()
    {
        return $this->hasMany(config('vtlabs_media.models.media_content'), 'media_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function assignor()
    {
        return $this->belongsTo(User::class);
    }

    public function assignee()
    {
        return $this->belongsTo(User::class);
    }

    public function parent()
    {
        return $this->belongsTo(config('vtlabs_media.models.media'), 'parent_media_id');
    }
}
