<?php

namespace Vtlabs\Media\Jobs;

use GuzzleHttp\Client;
use Illuminate\Bus\Queueable;
use Vtlabs\Media\Models\Media;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class QuickbaseWebhook implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected Media $media;

    // Recommended: Set higher timeouts for jobs involving video processing
    public int $timeout = 30;
    public int $tries = 3;

    /**
     * Create a new job instance.
     *
     * @param Media $media The media model instance to moderate.
     * @param float $minConfidence The minimum confidence level for moderation labels (0-100).
     * @param string $textContentLanguageCode The language code for text content (e.g., 'en', 'es').
     */
    public function __construct(Media $media)
    {
        $this->media = $media;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(): void
    {
        $media = $this->media->fresh() ?? $this->media;

        $client = new Client();
        
        $settings = CoreHelper::settingsAsDictionary();

        if(!isset($settings['webhook_url'])) {
            Log::error("Quickbase webook url not defined");
            throw new \Exception("Quickbase webook url not defined");
        }
        
        $url = $settings['webhook_url'];

        $imageUrls = [];

        foreach($media->media as $mediaItem) {
            $imageUrls[] = $mediaItem->getFullUrl();
        }

        $payload = [
            'id' => $media->id,
            'title' => $media->title,
            'category_id' => $media->category->id,
            'address' => $media->address,
            'latitude' => $media->latitude,
            'longitude' => $media->longitude,
            'status' => $media->status,
            'user_id' => $media->user_id,
            'assignor_id' => $media->assignor_id,
            'assignee_id' => $media->assignee_id,
            'image_urls' => $imageUrls,
            'admin_id' => $media->admin_id,
            'created_at' => $media->created_at,
            'updated_at' => $media->updated_at,
        ];

        try {
            // Make the POST request
            $response = $client->post($url, [
                'json' => $payload, // 'json' sends the payload as JSON, 'form_params' sends as form-urlencoded
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'timeout' => 20, // Optional: timeout after 10 seconds
                'connect_timeout' => 60, // Optional: connection timeout
            ]);

            $statusCode = $response->getStatusCode();
            $responseBody = $response->getBody()->getContents();

            if ($statusCode >= 200 && $statusCode < 300) {
                Log::info('Quickbase webhook sent successfully.', [
                    'status' => $statusCode,
                    'response' => $responseBody,
                    'payload' => $payload
                ]);
            } else {
                // Quickbase returned a non-2xx status code
                Log::error('Quickbase webhook failed with non-success status.', [
                    'status' => $statusCode,
                    'response' => $responseBody,
                    'payload' => $payload
                ]);
            }
        } catch (RequestException $e) {
            // Catch Guzzle-specific exceptions (e.g., network errors, 4xx/5xx responses)
            Log::error('Quickbase webhook request failed (Guzzle exception).', [
                'error_message' => $e->getMessage(),
                'request_url' => $url,
                'payload' => $payload,
                'response' => $e->hasResponse() ? $e->getResponse()->getBody()->getContents() : 'No response body'
            ]);
        } catch (\Exception $e) {
            // Catch any other general exceptions
            Log::error('Quickbase webhook encountered an unexpected error.', [
                'error_message' => $e->getMessage(),
                'request_url' => $url,
                'payload' => $payload
            ]);
        }
    }
}
