<?php

namespace Vtlabs\Media\Jobs;

use Illuminate\Support\Str;
use Illuminate\Bus\Queueable;
use Vtlabs\Media\Models\Media;
use Aws\Exception\AwsException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Aws\Rekognition\RekognitionClient;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Aws\Comprehend\ComprehendClient; // Added for Comprehend

class ModerateMediaContent implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected Media $media;
    protected RekognitionClient $rekognitionClient;
    protected ComprehendClient $comprehendClient; // Added Comprehend client
    protected float $minConfidence;
    protected string $textContentLanguageCode; // Language code for text moderation

    // Recommended: Set higher timeouts for jobs involving video processing
    public int $timeout = 1800; // 30 minutes
    public int $tries = 3;

    /**
     * Create a new job instance.
     *
     * @param Media $media The media model instance to moderate.
     * @param float $minConfidence The minimum confidence level for moderation labels (0-100).
     * @param string $textContentLanguageCode The language code for text content (e.g., 'en', 'es').
     */
    public function __construct(Media $media, float $minConfidence = 75.0, string $textContentLanguageCode = 'en')
    {
        $this->media = $media;
        $this->minConfidence = $minConfidence;
        $this->textContentLanguageCode = $textContentLanguageCode;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(): void
    {
        Log::info("Starting media content moderation for media ID: {$this->media->id} with min confidence: {$this->minConfidence}");
        
        $this->media = $this->media->fresh() ?? $this->media;

        $awsConfig = [
            'version' => 'latest',
            'region' => env('AWS_DEFAULT_REGION', 'us-east-1'),
        ];
        $this->rekognitionClient = new RekognitionClient($awsConfig);
        $this->comprehendClient = new ComprehendClient($awsConfig); // Initialize Comprehend client

        $overallModerationStatus = 'accepted';
        $temporaryS3VideoPath = null;
        $s3VideoBucket = env('AWS_BUCKET');

        try {
            // 1. Image Moderation
            foreach ($this->media->media as $imageMedia) {
                $imageUrl = $imageMedia->original_url;

                if ($imageUrl) {
                    Log::info("Starting image moderation for media ID: {$this->media->id}, URL: {$imageUrl}");
                    $imageParams = [];

                    if (Str::startsWith($imageUrl, 's3://')) {
                        list($bucket, $key) = $this->parseS3Url($imageUrl);
                        if ($bucket && $key) {
                            $imageParams = ['S3Object' => ['Bucket' => $bucket, 'Name' => $key]];
                        } else {
                            Log::error("Invalid S3 image_url for media ID: {$this->media->id}. URL: {$imageUrl}. Image moderation skipped.");
                        }
                    } elseif (Str::startsWith($imageUrl, ['http://', 'https://'])) {
                        try {
                            $imageContents = Http::timeout(60)->get($imageUrl)->body();
                            if (empty($imageContents)) {
                                Log::error("Downloaded image content is empty for media ID: {$this->media->id}. URL: {$imageUrl}.");
                            } else {
                                $imageParams = ['Bytes' => $imageContents];
                                Log::info("Image downloaded from HTTPS URL for media ID: {$this->media->id}");
                            }
                        } catch (\Throwable $e) {
                            Log::error("Failed to download image from URL: {$imageUrl} for media ID: {$this->media->id}. Error: {$e->getMessage()}");
                        }
                    } else {
                        Log::error("Unsupported image_url scheme for media ID: {$this->media->id}. URL: {$imageUrl}. Image moderation skipped.");
                    }

                    if (!empty($imageParams)) {
                        $imageModerationResult = $this->rekognitionClient->detectModerationLabels([
                            'Image' => $imageParams,
                            'MinConfidence' => $this->minConfidence,
                        ]);

                        Log::info("Image moderation result: {result}", ['result' => json_encode($imageModerationResult)]);

                        if (!empty($imageModerationResult['ModerationLabels'])) {
                            Log::warning("Image moderation labels found for media ID: {$this->media->id}", ['labels' => $imageModerationResult['ModerationLabels']]);
                            $overallModerationStatus = 'rejected';
                        } else {
                            Log::info("Image moderation clear for media ID: {$this->media->id}");
                        }
                    }
                }
            }

            // 2. Video Moderation (only if overall status is still 'accepted' and video_url exists)
            if ($overallModerationStatus === 'accepted' && $this->media->content) {
                foreach ($this->media->content as $video) {
                    Log::info("Starting video moderation for media ID: {$this->media->id}, URL: {$video->original_source}");
                    $videoS3Object = null;

                    if (Str::startsWith($video->original_source, 's3://')) {
                        list($bucket, $key) = $this->parseS3Url($video->original_source);
                        if ($bucket && $key) {
                            $videoS3Object = ['Bucket' => $bucket, 'Name' => $key];
                            $s3VideoBucket = $bucket;
                        } else {
                            Log::error("Invalid S3 video_url for media ID: {$this->media->id}. URL: {$video->original_source}. Video moderation skipped.");
                        }
                    } elseif (Str::startsWith($video->original_source, ['http://', 'https://'])) {
                        if (empty($s3VideoBucket)) {
                            Log::error("AWS_BUCKET environment variable is not set. Cannot upload temporary video for media ID: {$this->media->id}. Video moderation skipped.");
                        } else {
                            try {
                                Log::info("Downloading video from URL: {$video->original_source} for media ID: {$this->media->id}");
                                $videoContents = Http::timeout(600)->get($video->original_source)->body();
                                if (empty($videoContents)) {
                                    Log::error("Downloaded video content is empty for media ID: {$this->media->id}. URL: {$video->original_source}.");
                                } else {
                                    $originalFileName = basename(parse_url($video->original_source, PHP_URL_PATH));
                                    $extension = pathinfo($originalFileName, PATHINFO_EXTENSION);
                                    $safeFileName = Str::slug(pathinfo($originalFileName, PATHINFO_FILENAME)) ?: 'video';
                                    $temporaryS3VideoPath = 'rekognition-temp-videos/' . $this->media->id . '/' . $safeFileName . '_' . Str::uuid()->toString() . ($extension ? '.' . $extension : '');

                                    Storage::disk('s3')->put($temporaryS3VideoPath, $videoContents);
                                    $videoS3Object = ['Bucket' => $s3VideoBucket, 'Name' => $temporaryS3VideoPath];
                                    Log::info("Video downloaded and uploaded to temporary S3 location: s3://{$s3VideoBucket}/{$temporaryS3VideoPath} for media ID: {$this->media->id}");
                                }
                            } catch (\Throwable $e) {
                                Log::error("Failed to download or upload video from URL: {$video->original_source} for media ID: {$this->media->id}. Error: {$e->getMessage()}");
                            }
                        }
                    } else {
                        Log::error("Unsupported video_url scheme for media ID: {$this->media->id}. URL: {$video->original_source}. Video moderation skipped.");
                    }

                    if ($videoS3Object) {
                        $startJobResult = $this->rekognitionClient->startContentModeration([
                            'Video' => ['S3Object' => $videoS3Object],
                            'MinConfidence' => $this->minConfidence,
                        ]);

                        $rekognitionJobId = $startJobResult['JobId'];
                        Log::info("Video moderation job started for media ID: {$this->media->id}. Rekognition JobId: {$rekognitionJobId}");

                        $videoModerationOutcome = $this->pollForVideoModerationResults($rekognitionJobId);

                        if ($videoModerationOutcome === 'rejected') {
                            $overallModerationStatus = 'rejected';
                        } elseif ($videoModerationOutcome === 'failed' || $videoModerationOutcome === 'timeout') {
                            // $this->media->status = 'moderation_' . $videoModerationOutcome;
                            // $this->media->save();
                            Log::error("Video moderation {$videoModerationOutcome} for media ID: {$this->media->id}. Rekognition JobId: {$rekognitionJobId}");
                            if ($temporaryS3VideoPath && Str::startsWith($video->original_source, ['http://', 'https://'])) {
                                $this->cleanupTemporaryS3File($s3VideoBucket, $temporaryS3VideoPath, "after video moderation {$videoModerationOutcome}");
                            }
                            return;
                        }
                    }
                }
            }

            // 3. Text Moderation (only if overall status is still 'accepted' and text exists)
            if ($overallModerationStatus === 'accepted' && !empty(trim($this->media->title ?? ''))) {
                Log::info("Starting text moderation for media ID: {$this->media->id}");
                try {
                    $textModerationResult = $this->comprehendClient->detectToxicContent([
                        'TextSegments' => [['Text' => $this->media->title]],
                        'LanguageCode' => $this->textContentLanguageCode,
                    ]);

                    $foundToxicContent = false;
                    Log::info("Text moderation result {$textModerationResult}");
                    if (!empty($textModerationResult['ResultList'])) {
                        foreach ($textModerationResult['ResultList'] as $segmentResult) {
                            if (!empty($segmentResult['Labels'])) {
                                foreach ($segmentResult['Labels'] as $label) {
                                    // Comprehend scores are 0-1, minConfidence is 0-100
                                    if (isset($label['Score']) && ($label['Score'] * 100.0) >= $this->minConfidence) {
                                        Log::warning("Text moderation label found for media ID: {$this->media->id}", [
                                            'label' => $label['Name'],
                                            'score' => $label['Score'],
                                            'text_segment' => Str::limit($this->media->title, 100)
                                        ]);
                                        $foundToxicContent = true;
                                        break 2; // Break both inner and outer loop for labels
                                    }
                                }
                            }
                        }
                    }

                    if ($foundToxicContent) {
                        $overallModerationStatus = 'rejected';
                        Log::warning("Text content rejected for media ID: {$this->media->id}");
                    } else {
                        Log::info("Text moderation clear for media ID: {$this->media->id}");
                    }
                } catch (AwsException $e) {
                    Log::error("AWS Comprehend API error for media ID: {$this->media->id}. Message: {$e->getMessage()}", [
                        'aws_error_code' => $e->getAwsErrorCode(),
                        'aws_error_type' => $e->getAwsErrorType(),
                        'trace_small' => Str::limit($e->getTraceAsString(), 500),
                    ]);
                    // Decide if a Comprehend error should fail the whole job or just skip text moderation
                    // For now, let's log and continue, potentially accepting if other content is fine.
                    // To make it stricter, you could set $overallModerationStatus = 'moderation_error';
                } catch (\Throwable $e) {
                    Log::error("General error during text moderation for media ID: {$this->media->id}. Message: {$e->getMessage()}", [
                        'trace_small' => Str::limit($e->getTraceAsString(), 500),
                    ]);
                }
            } elseif ($overallModerationStatus === 'accepted' && empty(trim($this->media->title ?? ''))) {
                Log::info("No text content to moderate for media ID: {$this->media->id}");
            }


            $nextStatus = 'rejected';
            if ($overallModerationStatus == 'accepted') {
                $nextStatus = 'open';
            }
            $this->media->setStatus($nextStatus);
            $this->media->save();
            Log::info("Media ID: {$this->media->id} processed. Final status: {$this->media->status}");

            if($nextStatus == 'open') {
                $this->media->user->sendPushNotification(
                    'customer',
                __('vtlabs_media::messages.notification_post_accepted_title'),
                    __('vtlabs_media::messages.notification_post_accepted_body'),
                    ['media_id' => $this->media->id]
                );
            } else if($nextStatus == 'rejected') {
                    $this->media->user->sendPushNotification(
                        'customer',
                    __('vtlabs_media::messages.notification_post_rejected_title'),
                        __('vtlabs_media::messages.notification_post_rejected_body'),
                        ['media_id' => $this->media->id]
                    );
            }

        } catch (AwsException $e) { // Catch Rekognition specific or other AWS SDK issues not caught above
            Log::error("AWS SDK Exception during media moderation for media ID: {$this->media->id}. Message: {$e->getMessage()}", [
                'aws_error_code' => $e->getAwsErrorCode(),
                'aws_error_type' => $e->getAwsErrorType(),
                'trace_small' => Str::limit($e->getTraceAsString(), 500),
            ]);
            // $this->media->status = 'moderation_error';
            // $this->media->save();
            throw $e;
        } catch (\Throwable $e) {
            Log::error("General error during media moderation for media ID: {$this->media->id}. Message: {$e->getMessage()}", [
                'trace_small' => Str::limit($e->getTraceAsString(), 1000),
            ]);
            // $this->media->status = 'moderation_error';
            // $this->media->save();
            throw $e;
        } finally {
            if ($temporaryS3VideoPath && Str::startsWith($video->original_source, ['http://', 'https://'])) {
                $this->cleanupTemporaryS3File($s3VideoBucket, $temporaryS3VideoPath, "in final cleanup");
            }
        }
    }

    /**
     * Polls for video moderation results.
     * (This method remains largely the same as before)
     * @param string $rekognitionJobId The JobId from AWS Rekognition.
     * @return string 'accepted', 'rejected', 'failed', or 'timeout'.
     */
    protected function pollForVideoModerationResults(string $rekognitionJobId): string
    {
        $videoModerationComplete = false;
        $maxAttempts = 120;
        $attempt = 0;
        $pollingIntervalSeconds = 6;

        while (!$videoModerationComplete && $attempt < $maxAttempts) {
            sleep($pollingIntervalSeconds);
            $attempt++;
            Log::info("Polling video moderation status for media ID: {$this->media->id}, Rekognition JobId: {$rekognitionJobId}, Attempt: {$attempt}");

            try {
                $statusResult = $this->rekognitionClient->getContentModeration([
                    'JobId' => $rekognitionJobId,
                    'SortBy' => 'TIMESTAMP',
                ]);
            } catch (AwsException $e) {
                Log::error("Error polling video moderation status for media ID: {$this->media->id}, JobId: {$rekognitionJobId}. Error: {$e->getMessage()}");
                if ($attempt >= $maxAttempts / 2) {
                    return 'failed';
                }
                continue;
            }

            $jobStatus = $statusResult['JobStatus'] ?? 'UNKNOWN';
            Log::info("Video moderation status for media ID: {$this->media->id}, Rekognition JobId: {$rekognitionJobId}: {$jobStatus}");

            if ($jobStatus === 'SUCCEEDED') {
                $videoModerationComplete = true;
                if (!empty($statusResult['ModerationLabels'])) {
                    Log::warning("Video moderation labels found for media ID: {$this->media->id}, Rekognition JobId: {$rekognitionJobId}", ['labels' => $statusResult['ModerationLabels']]);
                    return 'rejected';
                } else {
                    Log::info("Video moderation clear for media ID: {$this->media->id}, Rekognition JobId: {$rekognitionJobId}");
                    return 'accepted';
                }
            } elseif ($jobStatus === 'FAILED') {
                $videoModerationComplete = true;
                Log::error("Video moderation failed for media ID: {$this->media->id}, Rekognition JobId: {$rekognitionJobId}. Reason: {$statusResult['StatusMessage']}");
                return 'failed';
            }
        }

        if (!$videoModerationComplete) {
            Log::error("Video moderation timed out after {$attempt} attempts for media ID: {$this->media->id}. Rekognition JobId: {$rekognitionJobId}");
            return 'timeout';
        }

        return 'failed'; // Fallback
    }

    /**
     * Parses an S3 URL (s3://bucket/key) into bucket and key.
     *
     * @param string $s3Url The S3 URL.
     * @return array{0: string|null, 1: string|null} An array containing bucket and key, or [null, null] if parsing fails.
     */
    private function parseS3Url(string $s3Url): array
    {
        if (preg_match('/^s3:\/\/([^\/]+)\/(.+)$/', $s3Url, $matches)) {
            return [$matches[1], $matches[2]];
        }
        Log::warning("Failed to parse S3 URL: {$s3Url}. Expected format: s3://bucket-name/key-name.ext");
        return [null, null];
    }

    /**
     * Cleans up a temporary S3 file.
     *
     * @param string|null $bucket
     * @param string|null $path
     * @param string $context For logging purposes
     * @return void
     */
    private function cleanupTemporaryS3File(?string $bucket, ?string $path, string $context = "cleanup"): void
    {
        if ($bucket && $path) {
            Log::info("Attempting to delete temporary S3 file: s3://{$bucket}/{$path} for media ID: {$this->media->id} ({$context})");
            try {
                Storage::disk('s3')->delete($path);
                Log::info("Successfully deleted temporary S3 file: s3://{$bucket}/{$path}");
            } catch (\Throwable $deleteException) {
                Log::error("Failed to delete temporary S3 file: s3://{$bucket}/{$path}. Error: {$deleteException->getMessage()}");
            }
        }
    }

    /**
     * The job failed to process.
     * (This method remains the same as before)
     * @param  \Throwable  $exception
     * @return void
     */
    public function failed(\Throwable $exception): void
    {
        Log::critical("Job ModerateMediaContent has permanently failed for media ID: {$this->media->id}. Reason: {$exception->getMessage()}");

        $media = Media::find($this->media->id);
        if ($media) {
            // $media->status = 'moderation_job_failed';
            // $media->save();
        } else {
            Log::error("Could not find media with ID {$this->media->id} to mark as moderation_job_failed.");
        }
    }
}
