<?php

namespace Vtlabs\Media\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Media\Models\Media;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Ecommerce\Models\Coupon;
use Vtlabs\Media\Models\MediaOrder;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Media\Filters\MediaOrderFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Http\Resources\Media\MediaOrderResource;

/**
 * @group  Media
 *
 * APIs for media order
 */
class MediaOrderController extends Controller
{
    private $settings;

    public function __construct()
    {
        $this->settings = CoreHelper::settingsAsDictionary();
    }

    public function index(Request $request)
    {
        $request->validate([
            'paid' => 'sometimes'
        ]);

        $request->merge([
            'user' => Auth::id()
        ]);

        $mediaOrders = MediaOrder::filter($request->all(), MediaOrderFilter::class)->ordered();

        return MediaOrderResource::collection($mediaOrders->paginate());
    }

    /**
     * Purchase media
     */
    public function store(Media $media, Request $request)
    {
        $request->validate([
            'coupon_code' => 'sometimes',
            'meta' => 'sometimes|json',
            'payment_method_slug' => 'required|exists:payment_methods,slug'
        ]);

        if($media->isPurchased) {
            return response(["message" => 'Already purchased'], 400);
        }

        $subtotal = $media->price;
        $taxes = $this->calculateTax($subtotal);
        $discount = $this->calculateDiscount($request->coupon_code, $subtotal);
        $total = ($subtotal + $taxes) - $discount;

        // create an order instance
        $order = MediaOrder::create([
            'subtotal' => $subtotal,
            'taxes' => $taxes,
            'discount' => $discount,
            'total' => $total,
            'user_id' => Auth::id(),
            'media_id' => $media->id,
            'meta' => !empty($request->meta) ? json_decode($request->meta, true) : null
        ]);

        // payment
        $paymentMethodSlug = $request->payment_method_slug;
        $payment = Auth::user()->createPayment($order, $total, $paymentMethodSlug);

        // if media is free, mark the payment as paid
        if($media->price == 0) {
            $payment->setStatus('paid');
            $payment->save();
        }

        return new MediaOrderResource($order->fresh());
    }

    public function update(MediaOrder $mediaOrder, Request $request)
    {
        $request->validate([
            'meta' => 'sometimes|json'
        ]);

        if($request->meta) {
            $mediaOrder->meta = json_decode($request->meta, true);
        }

        return new MediaOrderResource($mediaOrder->fresh());
    }

    private function calculateTax($amount)
    {
        $tax = $this->settings['tax_in_percent'] ?? 0;
        return ($amount * $tax) / 100;
    }

    private function calculateDiscount($coupon_code, $amount)
    {
        $discount = 0;

        if (!isset($coupon_code)) {
            return 0;
        }

        try {
            $coupon = Coupon::getCouponFromCode($coupon_code, $this->user);
            if ($coupon !== null) {
                if ($coupon->type == 'fixed') {
                    $discount = $coupon->reward;
                }

                if ($coupon->type == 'percent') {
                    $discount = ($amount * $coupon->reward) / 100;
                }

                $coupon->users()->attach($this->user->id, [
                    'used_at' => Carbon::now(),
                ]);
            }
        } catch (\Exception $ex) {
            //
        }

        return $discount;
    }
}
