<?php

namespace Vtlabs\Media\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Overtrue\LaravelLike\Like;
use Vtlabs\Media\Models\Media;
use Vtlabs\Media\Models\Comment;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Core\Models\Notification;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Media\Filters\MediaFilter;
use Vtlabs\Media\Models\MediaContent;
use Vtlabs\Media\Models\MediaEpisode;
use Illuminate\Support\Facades\Storage;
use Vtlabs\Media\Jobs\ModerateMediaContent;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Core\Http\Resources\UserResource;
use Vtlabs\Media\Filters\MediaEpisodeFilter;
use Vtlabs\Core\Helpers\PushNotificationHelper;
use Vtlabs\Media\Http\Resources\Media\MediaResource;
use Vtlabs\Media\Http\Resources\Media\CommentResource;
use Vtlabs\Media\Http\Resources\Media\MediaRatingResource;
use Vtlabs\Media\Http\Resources\Media\MediaEpisodeResource;

/**
 * @group  Media
 *
 * APIs for media
 */
class MediaController extends Controller
{
    public function __construct()
    {
        // since this url can be used with or without Authorization, we are applying auth middleware,
        // if Authorization header was present, so that we can get current user using Auth::user()
        if (array_key_exists('HTTP_AUTHORIZATION', $_SERVER)) {
            $this->middleware('auth:api');
        }
    }

    /**
     * List of media
     *
     * @queryParam  search string Search media.
     * @queryParam  recent boolean Recent media items. Example: 1
     * @queryParam  category string Media by category id.
     * @queryParam  author string Media by author id.
     * 
     * @responseFile responses/media/media.list.get.200.json
     */
    public function index(Request $request)
    {
        $request->validate([
            'search' => 'sometimes',
            'recent' => 'sometimes|boolean',
            'category' => 'sometimes',
            'categorizable' => 'sometimes|boolean',
            'following' => 'sometimes|boolean',
            'author' => 'sometimes|exists:authors,id',
            'user' => 'sometimes|exists:users,id',
            'liked_by' => 'sometimes|exists:users,id',
            'is_parent' => 'sometimes|boolean'
        ]);

        if(Auth::check() && Auth::user()->hasRole('worker')) {
            $request->merge([
                'assignee' => Auth::id()
            ]);
        }

        $mediaItems = Media::filter($request->all(), MediaFilter::class);

        return MediaResource::collection($mediaItems->paginate());
    }

    /**
     * Media by ID
     * 
     * @responseFile responses/media/media.show.get.200.json
     */
    public function show(Media $media)
    {
        $media->save();
        return new MediaResource($media);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'sometimes',
            'meta' => 'sometimes|json|nullable',
            'category_id' => 'sometimes|exists:categories,id',
            'image_urls' => 'sometimes|array',
            'content' => "sometimes|nullable|array",
            'content.*.original_source' => 'required',
            'address' => 'sometimes',
            'longitude' => 'sometimes',
            'latitude' => 'sometimes',
            'admin_id' => 'required',
        ]);

        request()->merge([
            "user_id" => Auth::id()
        ]);

        // if worker is creating a post, we will auto assign the post to him
        if(Auth::user()->hasRole('worker')) {
            $request->merge([
                'assignee_id' => Auth::id()
            ]);
        }

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        // save hastags
        Media::saveHashtags($request->title);

        $media = Media::create($request->only(['title','meta','user_id','category_id', 'address', 'longitude', 'latitude', 'is_public', 'assignee_id', 'admin_id']));

        $media->save();

        // images
        if (!empty($request->image_urls)) {
            foreach ($request->image_urls as $image) {
                $media->addMediaFromUrl($image)->toMediaCollection("images");
            }
        }

        // content
        if (!empty($request->content)) {
            $mediaContent = [];
            foreach ($request->content as $content) {
                $mediaContent[] = new MediaContent([
                    'original_source' => $content['original_source'],
                    'type' => 'main',
                    'source' => ''
                ]);
            }
            $media->content()->saveMany($mediaContent);
        }

        ModerateMediaContent::dispatch($media, env('MODERATE_MEDIA_THRESHOLD', 20));

        return new MediaResource($media->fresh());
    }

    public function update(Media $media, Request $request)
    {
        $request->validate([
            'image_urls' => 'sometimes|array',
            'after_assets.*' => 'sometimes',
            'status' => 'sometimes'
            
        ]);

        if (!empty($request->image_urls)) {
            foreach ($request->image_urls as $image) {
                $media->addMediaFromUrl($image)->toMediaCollection("after_images");
            }
        }

        if($request->after_assets) {
            foreach ($request->after_assets as $key => $afterAsset) {
                if ($key >= 0 && $key <= 2) {
                    $media->{'after_assets_url_' . ($key + 1)} = $afterAsset;
                }
            }
            $media->save();
        }

        if($request->status) {
            $media->setStatus($request->status);
        }

        return new MediaResource($media->fresh());
    }

    public function updateCount(Media $media, Request $request)
    {
        $request->validate([
            'shares_count' => 'sometimes|boolean',
            'views_count' => 'sometimes|boolean',
        ]);

        if ($request->shares_count) {
            $media->shares_count++;
            $media->save();
        }

        if ($request->views_count) {
            $media->views_count++;
            $media->save();
        }

        return response()->json((object)[]);
    }

    public function bulkUpdateViewCount(Request $request)
    {
        $request->validate([
            'ids' => 'required|array'
        ]);

        Media::whereIn('id', $request->ids)->increment('views_count');

        return response()->json((object)[]);
    }

    public function destroy(Media $media)
    {
        // update hashtags frequency
        Media::deleteHashtags($media->title);

        $media->delete();

        return response()->json((object)[], 200);
    }

    /**
     * Favourite Media list
     * 
     * @authenticated
     * 
     * @responseFile responses/media/media.list.get.200.json
     */
    public function favourites(Request $request)
    {
        $request->validate([
            'category' => 'sometimes|nullable'
        ]);

        // apply category filter

        $mediaList = Auth::user()->favorites()->with('favoriteable')->where('favoriteable_type', Media::class)->orderByDesc('created_at')->get()->pluck(['favoriteable']);

        // remove null values
        $mediaList = $mediaList->filter();

        if ($request->category) {
            $categoryId = $request->category;
            $mediaList = $mediaList->filter(function ($media, $key) use ($categoryId) {
                // compare id or slug of the category
                return $media->category->id == $categoryId || $media->category->slug == $categoryId;
            });
        }

        return MediaResource::collection($mediaList);
    }

    /**
     * Toggle Favourite
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleFavourite(Media $media, Request $request)
    {
        $media->toggleFavorite();

        return response()->json((object)[]);
    }

    public function favouriteMediaImages()
    {
        // api to get images of favourite media, used to show media images in UI sections
        $favMedias = Auth::user()->favorites()->where('favoriteable_type', Media::class)->orderByDesc('created_at')->limit(4)->get()->pluck(['favoriteable_id'])->toArray();
        $medias = Media::whereIn('id', $favMedias)->get();

        $images = [];

        foreach ($medias as $media) {
            array_push($images, $media->getMediaUrlsAttribute()["images"][0]["default"]);
        }

        return response()->json(['media_images' => $images]);
    }

    /**
     * List of episodes
     *
     * @queryParam  season int Season number
     * 
     * @responseFile responses/media/episode.list.get.200.json
     */
    public function episodes(Media $media, Request $request)
    {
        $request->validate([
            'season' => 'sometimes|number'
        ]);

        $episodes = MediaEpisode::where('parent_media_id', $media->id)->filter($request->all(), MediaEpisodeFilter::class)->ordered();
        return MediaEpisodeResource::collection($episodes->get());
    }

    /**
     * Rating Media list
     * 
     * @responseFile responses/media/rating.list.get.200.json
     */
    public function ratingList(Media $media, Request $request)
    {
        return MediaRatingResource::collection($media->raters(User::class)->paginate());
    }

    /**
     * Rate media
     * 
     * @authenticated
     * 
     * @bodyParam  rating float required Rating in number
     * @bodyParam  review string required Rating in text
     * 
     * @response []
     */
    public function ratingStore(Media $media, Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric',
            'review' => 'required',
            'created_at' => Carbon::now()
        ]);

        $user = Auth::user();

        $user->unrate($media);
        $user->rate($media, $request->rating, $request->review);

        return response()->json((object)[], 200);
    }

    /**
     * Get rating summary
     * 
     * @responseFile responses/media/rating.summary.get.200.json
     */
    public function ratingSummary(Media $media)
    {
        return response()->json([
            "average_rating" => $media->averageRating(User::class),
            "total_ratings" => $media->raters(User::class)->count(),
            "summary" => DB::table('ratings')->selectRaw('count(*) as total, ROUND(rating) as rounded_rating')
                ->where('rateable_type', Media::class)
                ->where('rateable_id', $media->id)
                ->where('rater_type', User::class)
                ->groupBy('rounded_rating')
                ->get()
        ]);
    }

    /**
     * Toggle Like
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleLike(Media $media, Request $request)
    {
        Auth::user()->toggleLike($media);

        $liked = Auth::user()->hasLiked($media);

        if ($liked && $media->user_id != Auth::id()) {
            Notification::create([
                'type' => 'like',
                'text' => 'liked your post',
                'meta' => ['media' => new MediaResource($media)],
                'user_id' => $media->user_id,
                'from_user_id' => Auth::id()
            ]);

            $media->user->sendPushNotification(
                'customer',
                __('vtlabs_media::messages.notification_like_post_title', ['liked_by' => Auth::user()->name]),
                __('vtlabs_media::messages.notification_like_post_body'),
                ['media_id' => $media->id]
            );
        }

        return response()->json(["like" => $liked]);
    }

    /**
     * List of likers
     * 
     * @authenticated
     * 
     * @response []
     */
    public function likers(Media $media, Request $request)
    {
        return UserResource::collection($media->likers()->paginate());
    }

    /**
     * List of liked media
     * 
     * @authenticated
     * 
     * @response []
     */
    public function likes(Request $request)
    {
        $likes = Like::withType('Vtlabs\Media\Models\Media')->where('user_id', Auth::id())->get()->pluck(['likeable_id'])->toArray();

        return MediaResource::collection(Media::whereIn($likes)->paginate());
    }

    /**
     * List of comments
     * 
     * @authenticated
     * 
     * @response []
     */
    public function listComments(Media $media, Request $request)
    {
        return CommentResource::collection($media->comments()->latest()->paginate());
    }

    public function listReplyComments(Comment $comment, Request $request)
    {
        return CommentResource::collection($comment->comments()->paginate());
    }

    /**
     * Post a comment
     * 
     * @authenticated
     * 
     * @response []
     */
    public function createComment(Media $media, Request $request)
    {
        $request->validate([
            'comment' => 'required|string'
        ]);

        $comment = $media->comment($request->comment);

        if ($media->user_id != Auth::id()) {
            Notification::create([
                'type' => 'comment',
                'text' => 'commented on your post',
                'meta' => ['media' => new MediaResource($media)],
                'user_id' => $media->user_id,
                'from_user_id' => Auth::id()
            ]);

            $media->user->sendPushNotification(
                'customer',
                __('vtlabs_media::messages.notification_comment_post_title', ['comment_by' => Auth::user()->name]),
                __('vtlabs_media::messages.notification_comment_post_body'),
                ['media_id' => $media->id]
            );
        }

        return new CommentResource($comment);
    }

    /**
     * Post a comment reply
     * 
     * @authenticated
     * 
     * @response []
     */
    public function replyComment(Comment $comment, Request $request)
    {
        $request->validate([
            'comment' => 'required|string'
        ]);

        $newcomment = $comment->comment($request->comment);

        return new CommentResource($newcomment);
    }

    /**
     * Toggle comment Like
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleCommentLike(Comment $comment, Request $request)
    {
        Auth::user()->toggleLike($comment);

        $liked = Auth::user()->hasLiked($comment);

        if ($liked && $comment->user_id != Auth::id()) {
            Notification::create([
                'type' => 'comment_like',
                'text' => 'liked your post',
                'meta' => ['comment' => new CommentResource($comment)],
                'user_id' => $comment->user_id,
                'from_user_id' => Auth::id()
            ]);

            $media = Media::find($comment->commentable_id);
            $media->user->sendPushNotification(
                'customer',
                __('vtlabs_media::messages.notification_like_comment_title', ['liked_by' => Auth::user()->name]),
                __('vtlabs_media::messages.notification_like_comment_body'),
                ['media_id' => $media->id]
            );
        }

        return response()->json(["like" => $liked]);
    }

    /**
     * List of likers
     * 
     * @authenticated
     * 
     * @response []
     */
    public function commentLikers(Comment $comment, Request $request)
    {
        return UserResource::collection($comment->likers()->paginate());
    }

    public function updateComment(Comment $comment, Request $request)
    {
        $request->validate([
            'comment' => 'required|string'
        ]);

        $comment->comment = $request->comment;

        $comment->save();

        return new CommentResource($comment);
    }

    public function deleteComment(Comment $comment, Request $request)
    {
        $comment->delete();

        return response()->json((object)[], 200);
    }

    public function stats(User $user, Request $request)
    {
        $idColumn = $user->hasRole('worker') ? 'assignee_id' : 'user_id';
        $id = $user->id;

        $medias = Media::where($idColumn, $id)->pluck('id')->toArray();

        return response()->json([
            'total_likes' => Like::withType('Vtlabs\Media\Models\Media')->whereIn('likeable_id', $medias)->count(),
            'total_opened' => Media::where($idColumn, $id)->currentStatus('open')->count(),
            'total_reviewed' => Media::where($idColumn, $id)->currentStatus('reviewed')->count(),
            'total_assigned' => Media::where($idColumn, $id)->currentStatus('assigned')->count(),
            'total_started' => Media::where($idColumn, $id)->currentStatus('started')->count(),
            'total_complete' => Media::where($idColumn, $id)->currentStatus('complete')->count(),
            'total_overdue' => Media::where($idColumn, $id)->currentStatus('overdue')->count(),
        ]);
    }

    public function repost(Media $media, Request $request)
    {
        $user = Auth::user();

        $repostMedia = $media->replicate()->fill([
            'description' => null,
            'short_description' => null,
            'views_count' => 0,
            'shares_count' => 0,
            'user_id' => $user->id,
            'meta' => array_merge($media->meta ?? [], ['original' => new MediaResource($media)]),
            'created_at' => now(),
            'updated_at' => now()
        ]);

        $repostMedia->save();

        Notification::create([
            'type' => 'repost',
            'text' => 'reposted your post',
            'meta' => [],
            'user_id' => $media->user_id,
            'from_user_id' => Auth::id()
        ]);

        $media->user->sendPushNotification(
            'customer',
            __('vtlabs_media::messages.notification_repost_title', ['repost_by' => Auth::user()->name]),
            __('vtlabs_media::messages.notification_repost_body'),
            ['media_id' => $media->id]
        );

        return new MediaResource($repostMedia->refresh());
    }

    public function report(Media $media, Request $request)
    {
        $request->validate([
            'reason' => 'sometimes'
        ]);

        $user = User::find(Auth::id());

        $media->report($user, ['reason' => $request->reason]);

        return response([], 200);
    }

    public function getPopularHashtags(Request $request)
    {
        $request->validate([
            'limit' => 'sometimes|numeric'
        ]);

        $limit = $request->limit ?? 10;

        return response()->json(DB::table('media_hashtags')->where('frequency', '>', 0)->orderByDesc('frequency')->limit($limit)->get());
    }

    public function notifyFollowers()
    {
        try {
            $user = Auth::user();

            $notifyIds = $user->followers->pluck(['notification'])->flatten()->filter()->toArray();

            $oneSignal = PushNotificationHelper::getOneSignalInstance('customer');

            $data['title'] = __('vtlabs_media::messages.notification_live_title', ['name' => $user->name]);
            $data['body'] = __('vtlabs_media::messages.notification_live_body', ['name' => $user->name]);

            $oneSignal->sendNotificationToUser(
                $data['title'],
                $notifyIds,
                null,
                array_merge($data, ['user_id' => $user->id])
            );
        } catch (\Exception $ex) {
            //
        }

        return response()->json((object)[]);
    }

    public function statusChart(Request $request)
    {
        $request->validate([
            'assignee' => 'required|exists:users,id',
            'duration' => 'required|in:hours,days,months,years',
            'status' => 'required |in:open,reviewed,assigned,started,complete,overdue'
        ]);

        $duration = $request->duration;
        $groupBy = "DATE('m.created_at')";
        $label = 'DATE(m.created_at) as created_at';
        $status = $request->status;

        switch ($duration) {
            case 'hours':
                $limit = 1;
                $groupBy = 'HOUR(m.created_at)';
                $label = 'HOUR(m.created_at) as created_at';
                break;
            case 'days':
                $limit = 7;
                $groupBy = 'DATE(m.created_at)';
                $label = 'DATE(m.created_at) as created_at';
                break;
            case 'months':
                $limit = 30;
                $groupBy = 'MONTH(m.created_at)';
                $label = 'MONTH(m.created_at) as created_at';
                break;
            case 'years':
                $limit = 365;
                $groupBy = 'YEAR(m.created_at)';
                $label = 'YEAR(m.created_at) as created_at';
                break;
        }

        $chartDataQuery = DB::table('media as m')->select(DB::raw($label), DB::raw('count(*) as total'))
            ->whereDate('m.created_at', '>', Carbon::now()->subDays($limit))
            ->whereDate('m.created_at', '<=', Carbon::now())
            ->join('statuses as s', 'm.id', '=', 's.model_id')
            ->where('s.model_type', '=', Media::class)
            ->groupBy(DB::raw($groupBy));

        
        $chartData = $chartDataQuery->where('s.name', $status)->get()->mapWithKeys(function ($item) {
            return [$item->created_at => $item->total];
        })->all();

        $chartLabel = array_map(function ($date) use ($duration) {
            if ($duration == 'week') {
                $in = date_create($date);
                return date_format($in, "D");
            }
            if ($duration == 'month') {
                $in = date_create($date);
                return date_format($in, "d");
            }
            if ($duration == 'year') {
                $in = date_create('2013-10-27'); // create a sample date object
                $in->setDate($in->format('Y'), $date, $in->format('d'));
                return $in->format("M");
            }
            if ($duration == 'all') {
                $in = date_create('2013-10-27'); // create a sample date object
                $in->setDate($date, $in->format('m'), $in->format('d'));
                return $in->format("Y");
            }
            return $date;
        }, $chartDataQuery->pluck('created_at')->toArray());

        return response()->json([
            "chart" => [
                "chartLabel" => $chartLabel,
                "linesData" => [
                    $chartData
                ]
            ]
        ]);
    }

    public function statusStats(Request $request)
    {
        // 1. Validate incoming request to ensure assignee ID is provided and valid
        $request->validate([
            'assignee' => 'required|exists:users,id'
        ]);

        $assigneeId = $request->assignee;

        // 2. Subquery to find the MAX(id) for each model_id (i.e., the latest status ID)
        // This is crucial for ensuring we only consider the most recent status for each media item.
        $latestStatusIds = DB::table('statuses')
            ->select('model_id', DB::raw('MAX(id) as max_status_id'))
            ->where('model_type', Media::class)
            ->groupBy('model_id');

        // 3. Main query to get the aggregated counts of the latest statuses
        $statsDataQuery = DB::table('media as m')
            // Join with the subquery to link media items to their latest status IDs
            ->joinSub($latestStatusIds, 'latest_s', function ($join) {
                $join->on('m.id', '=', 'latest_s.model_id');
            })
            // Join back to the statuses table using the latest status ID to get the status name
            ->join('statuses as s', function ($join) {
                $join->on('s.id', '=', 'latest_s.max_status_id')
                    ->where('s.model_type', Media::class); // Ensure correct model type
            })
            // Select the status name and count the occurrences
            ->select('s.name as status', DB::raw('count(*) as total'))
            // Filter by the requested assignee ID
            ->where('m.assignee_id', $assigneeId)
            // Group the results by the status name to get counts for each type
            ->groupBy('s.name')
            // Order by status name for consistent output
            ->orderBy('s.name');

        // 4. Execute the query and map the results
        $statsData = $statsDataQuery->get()->mapWithKeys(function ($item) {
            return [$item->status => $item->total];
        })->all();

        // 5. Return the aggregated status data as a JSON response
        return response()->json([
            "stats" => $statsData
        ]);
    }
}
