<?php

namespace Vtlabs\Media\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Media\Models\News;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Media\Filters\NewsFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Http\Resources\Media\Admin\NewsAdminResource;

class NewsController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean'
        ]);

        $news = News::filter($request->all(), NewsFilter::class);

        if ($request->pagination == '0') {
            $news = $news->get();
        } else {
            $news = $news->paginate();
        }
        return NewsAdminResource::collection($news);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'image' => 'sometimes|image',
            'sort_order' => 'required',
            'admin_id' => 'required|exists:users,id'
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        $news = News::create($request->only(['title', 'description', 'meta', 'sort_order', 'admin_id']));

        if ($request->image) {
            $news->addMedia($request->image)->toMediaCollection("images");
        }

        return new NewsAdminResource($news->fresh());
    }

    public function show($id)
    {
        return new NewsAdminResource(News::find($id));
    }

    public function update(News $news, Request $request)
    {
        $request->validate([
            'title' => 'required',
            'description' => 'required',
            'image' => 'sometimes|image',
            'sort_order' => 'required'
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        $news->fill($request->only(['title', 'description', 'meta', 'sort_order']));
        $news->save();

        if ($request->image) {
            $news->clearMediaCollection("images");
            $news->addMedia($request->image)->toMediaCollection("images");
        }

        return new NewsAdminResource($news->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        News::findOrFail($id)->delete();

        return response()->json([], 200);
    }
}
