<?php

namespace Vtlabs\Media\Http\Controllers\Api\Admin;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Vtlabs\Media\Models\Media;
use Vtlabs\Report\Models\Report;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Media\Filters\MediaFilter;
use Vtlabs\Media\Helpers\MediaHelper;
use Vtlabs\Media\Models\MediaContent;
use Illuminate\Support\Facades\Storage;
use Vtlabs\Media\Jobs\QuickbaseWebhook;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Http\Resources\Media\Admin\MediaAdminResource;
use Vtlabs\Media\Http\Resources\Media\Admin\ReportedMediaAdminResource;

class MediaController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'is_parent' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $medias = Media::filter($request->all(), MediaFilter::class);

        return MediaAdminResource::collection($medias->paginate());
    }

    public function show($id)
    {
        return new MediaAdminResource(Media::findOrFail($id));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'sometimes|translation',
            'description_translations' => 'sometimes|translation',
            'short_description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'status' => 'required|string',
            'is_public' => 'sometimes|integer',
            'user_id' => 'sometimes|exists:users,id',
            'assignor_id' => 'sometimes|exists:users,id',
            'assignee_id' => 'sometimes|exists:users,id',
            'category_id' => 'sometimes|exists:categories,id',
            'subcategories' => 'sometimes|array|exists:categories,id',
            'images.*' => 'sometimes|image',
            'videos.*' => 'sometimes|file|mimetypes:video/mp4,video/avi,video/quicktime',
            'address' => 'required|string',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
            'after_assets.*' => 'sometimes|file|mimetypes:video/mp4,video/avi,video/quicktime,image/jpeg,image/png,image/jpg',
            'admin_id' => 'required|exists:users,id'
        ]);

        request()->merge([
            "title" => request()->input('_title') ?? null,
            "description" => request()->input('_description') ?? null,
            "short_description" => request()->input('_short_description') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        $media = Media::create($request->only([
            'title',
            'description',
            'short_description',
            'meta',
            'category_id',
            'user_id',
            'address',
            'longitude',
            'latitude',
            'is_public',
            'assignor_id',
            'assignee_id',
            'admin_id'
        ]));

        $media->fill(['user_id' => Auth::id()]);

        $media->save();

        if ($request->status) {
            $media->setStatus($request->status);
        }

        // subcategories
        if (!empty($request->subcategories)) {
            $media->subcategories()->sync($request->subcategories);
        }

        // images
        if (!empty($request->images)) {
            foreach ($request->images as $image) {
                $media->addMedia($image)->toMediaCollection("images");
            }
        }

        // content
        if ($request->videos) {
            foreach ($request->videos as $video) {
                $extension = $video->getClientOriginalExtension();
                $uniqueName = Str::uuid() . '.' . $extension;
                $path = $video->storeAs('videos', $uniqueName, 'public');

                $mediaContent = MediaContent::create(['original_source' => Storage::url($path), 'type' => 'main', 'source' => '']);
            }
        }

        if ($request->after_assets) {
            foreach ($request->after_assets as $key => $afterAsset) {
                $extension = $afterAsset->getClientOriginalExtension();
                $uniqueName = Str::uuid() . '.' . $extension;
                $path = $afterAsset->storeAs('after_assets', $uniqueName, 'public');

                if ($key >= 0 && $key <= 2) {
                    $media->{'after_assets_url_' . ($key + 1)} = Storage::url($path);
                }
            }
            $media->save();
        }

        // if (!empty($request->content)) {
        //     foreach ($request->content as $content) {
        //         $meta = [];

        //         if (!empty($content['title'])) {
        //             $meta['title'] = $content['title'];
        //         }

        //         if (!empty($content['length'])) {
        //             $meta['length'] = $content['length'];
        //         }

        //         $mediaContent = MediaContent::create([
        //             'original_source' => $content['original_source'],
        //             'type' => $content['type'] ?? 'main',
        //             'meta' => !empty($meta) ? $meta : null,
        //             'source' => '',
        //             'media_id' => $media->id
        //         ]);
        //         if (!empty($content['image'])) {
        //             $mediaContent->addMedia($content['image'])->toMediaCollection("images");
        //         }
        //     }
        // }

        $media = $media->fresh();

        QuickbaseWebhook::dispatch($media);

        return new MediaAdminResource($media);
    }

    public function update($id, Request $request)
    {
        $media = Media::findOrFail($id);
        $assigneeUpdated = false;

        $request->validate([
            'title_translations' => 'required|translation',
            'description_translations' => 'sometimes|translation',
            'short_description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'status' => 'required|string',
            'is_public' => 'sometimes|integer',
            'user_id' => 'sometimes|exists:users,id',
            'assignor_id' => 'sometimes|exists:users,id',
            'assignee_id' => 'sometimes|exists:users,id',
            'category_id' => 'sometimes|exists:categories,id',
            'subcategories' => 'sometimes|array|exists:categories,id',
            'images.*' => 'sometimes|image',
            'videos.*' => 'sometimes|file|mimetypes:video/mp4,video/avi,video/quicktime',
            'address' => 'required|string',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
            'after_assets.*' => 'sometimes|file|mimetypes:video/mp4,video/avi,video/quicktime,image/jpeg,image/png,image/jpg'
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "description" => request()->input('_description') ?? null,
            "short_description" => request()->input('_short_description') ?? null
        ]);

        if ($request->assignee_id != $media->assignee_id) {
            $assigneeUpdated = true;
        }

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        $media->fill($request->only([
            'title',
            'description',
            'short_description',
            'meta',
            'category_id',
            'user_id',
            'address',
            'longitude',
            'latitude',
            'is_public',
            'assignor_id',
            'assignee_id'
        ]));

        $media->save();

        if ($request->status) {
            $media->setStatus($request->status);
        }

        // subcategories
        if (!empty($request->subcategories)) {
            $media->subcategories()->sync($request->subcategories);
        }

        // images
        if (!empty($request->images)) {
            $media->clearMediaCollection('images');
            foreach ($request->images as $image) {
                $media->addMedia($image)->toMediaCollection("images");
            }
        }

        if ($request->videos) {
            $media->content()->delete(); // delete existing content
            foreach ($request->videos as $video) {
                $extension = $video->getClientOriginalExtension();
                $uniqueName = Str::uuid() . '.' . $extension;
                $path = $video->storeAs('videos', $uniqueName, 'public');
                MediaContent::create(['original_source' => Storage::url($path), 'type' => 'main', 'source' => '', 'media_id' => $media->id]);
            }
        }

        if ($request->after_assets) {
            foreach ($request->after_assets as $key => $afterAsset) {
                $extension = $afterAsset->getClientOriginalExtension();
                $uniqueName = Str::uuid() . '.' . $extension;
                $path = $afterAsset->storeAs('after_assets', $uniqueName, 'public');

                if ($key >= 0 && $key <= 2) {
                    $media->{'after_assets_url_' . ($key + 1)} = Storage::url($path);
                }
            }
            $media->save();
        }

        if ($assigneeUpdated && $media->assignee_id) {
            Log::info("Send notification to assignee");
            $media->assignee->sendPushNotification(
                'customer',
                __('vtlabs_media::messages.notification_post_assigned_title'),
                __('vtlabs_media::messages.notification_post_assigned_title'),
                ['media_id' => $media->id]
            );
        }

        $media = $media->fresh();

        QuickbaseWebhook::dispatch($media);

        return new MediaAdminResource($media);
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        $media = Media::findOrFail($id);

        // update hashtags frequency
        Media::deleteHashtags($media->title);

        $media->delete();

        return response()->json([], 200);
    }

    // Get data for Media Status Distribution (Pie Chart).
    public function getMediaStatusDistribution(Request $request)
    {
        $request->validate([
            'admin_id' => 'sometimes'
        ]);

        // Step 1: Find the latest status ID for each media item in the 'statuses' table.
        // This ensures we're counting the *current* status for each media record.
        $latestStatusIds = DB::table('statuses as s')
            ->select('s.model_id', DB::raw('MAX(s.id) as latest_id'))
            ->join('media as m', 's.model_id', '=', 'm.id') // <-- INNER JOIN with the media table
            ->where('s.model_type', Media::class); // Filter statuses specifically for the Media model

        if ($request->admin_id) {
            $latestStatusIds = $latestStatusIds->where('m.admin_id', $request->admin_id);
        }

        $latestStatusIds =  $latestStatusIds->groupBy('s.model_id');

        // Step 2: Join the 'statuses' table with the subquery results (latest status IDs)
        // and then group by the status 'name' to get the counts for each current status.
        $statusCounts = DB::table('statuses as s_main')
            ->joinSub($latestStatusIds, 'latest_s', function ($join) {
                // Match on the latest status ID and the media item ID
                $join->on('s_main.id', '=', 'latest_s.latest_id')
                    ->on('s_main.model_id', '=', 'latest_s.model_id');
            })
            ->select('s_main.name as status', DB::raw('count(s_main.id) as count'))
            ->groupBy('s_main.name') // Group by the status name to get the total count for each status
            ->get();

        return response()->json($statusCounts);
    }

    // Get data for Media Status Changes Over Time (Line Chart)
    public function getMediaStatusChangesOverTime(Request $request)
    {
        $request->validate([
            'admin_id' => 'sometimes'
        ]);

        // Define the aggregation period. Defaults to 'month' for broader trends.
        // Accepts 'day', 'week', 'month', 'year' via query parameter.
        $period = $request->input('period', 'month');

        // Optional: Filter by start and end dates
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        // --- IMPORTANT UPDATE HERE ---
        // Join 'statuses' table with 'media' table to ensure we only count statuses
        // for currently existing Media models.
        $query = DB::table('statuses as s')
            ->join('media as m', 's.model_id', '=', 'm.id') // <-- INNER JOIN with the media table
            ->where('s.model_type', Media::class); // Ensure we only count statuses for Media models
        // and explicitly use 's.model_type' due to alias

        if ($request->admin_id) {
            $query->where('m.admin_id', $request->admin_id);
        }

        // Apply date filters if provided (now applied to 's.created_at')
        if ($startDate) {
            $query->where('s.created_at', '>=', $startDate);
        }
        if ($endDate) {
            $query->where('s.created_at', '<=', $endDate);
        }

        // Determine the database function for date formatting based on the requested period
        // Now using 's.created_at' due to the alias
        $dateSelect = DB::raw('DATE_FORMAT(s.created_at, \'%Y-%m\') as date'); // Default to month
        if ($period === 'day') {
            $dateSelect = DB::raw('DATE(s.created_at) as date');
        } elseif ($period === 'week') {
            $dateSelect = DB::raw('DATE_FORMAT(s.created_at, \'%X-%V\') as date'); // ISO week date format (YYYY-WW)
        } elseif ($period === 'year') {
            $dateSelect = DB::raw('YEAR(s.created_at) as date');
        }

        // Select the status name, the formatted date, and count occurrences
        // Now using 's.name' for status name
        $statusChangesRaw = $query->select(
            's.name as status', // <-- Use alias 's' for status name
            $dateSelect,
            DB::raw('count(*) as count')
        )
            ->groupBy('status', 'date') // Group by status and the formatted date
            ->orderBy('date') // Order by date to ensure chronological labels
            ->get();

        // --- Data Transformation for Line Chart ---
        // (This part remains unchanged as it processes the filtered data)
        $labels = [];
        $series = [];

        foreach ($statusChangesRaw as $item) {
            if (!in_array($item->date, $labels)) {
                $labels[] = $item->date;
            }
            if (!isset($series[$item->status])) {
                $series[$item->status] = [
                    'name' => $item->status,
                    'data' => []
                ];
            }
        }

        sort($labels);

        foreach ($series as $statusName => &$statusSeries) {
            $dataForStatus = [];
            foreach ($labels as $date) {
                $found = false;
                foreach ($statusChangesRaw as $item) {
                    if ($item->status === $statusName && $item->date === $date) {
                        $dataForStatus[] = $item->count;
                        $found = true;
                        break;
                    }
                }
                if (!$found) {
                    $dataForStatus[] = 0;
                }
            }
            $statusSeries['data'] = $dataForStatus;
        }

        $formattedData = [
            'labels' => $labels,
            'series' => array_values($series)
        ];

        return response()->json($formattedData);
    }

    // Get data for Media Count by Category (Bar Chart)
    public function getMediaCountByCategory(Request $request)
    {
        $request->validate([
            'admin_id' => 'required'
        ]);

        $adminId = $request->admin_id;

        // Join the 'media' table with the 'categories' table
        // Select the English title of the category and count media items for each
        $categoryCounts = Media::query()
            ->select(
                'categories.title->en as category_name', // Access the 'en' key from the JSON 'title' column
                DB::raw('count(media.id) as count')      // Count the media IDs
            )
            ->join('categories', 'media.category_id', '=', 'categories.id') // Join on category_id
            ->where('media.admin_id', $adminId)
            ->where('categories.admin_id', $adminId)
            // Group by both category ID and the category name to ensure correct aggregation,
            // especially important when selecting non-aggregated columns.
            ->groupBy('categories.id', 'category_name');

        if(MediaHelper::isSupervisor()) {
            $categoryCounts->whereIn('categories.id', MediaHelper::supervisorCategoryIds());
        }

        $categoryCounts = $categoryCounts
            ->orderBy('count', 'desc') // Optional: Order by count to show most popular categories first
            ->limit(15)
            ->get();

        return response()->json($categoryCounts);
    }

    public function getCompletionTimeMetrics(Request $request)
    {
        $request->validate([
            'admin_id' => 'required'
        ]);

        $adminId = $request->admin_id;

        // Step 1: Get the IDs of all Media items that are currently 'complete'.
        // We use the currentStatus scope, which you've confirmed works correctly
        // and handles existing media items.
        $completeMediaIds = Media::query()
            ->currentStatus('complete')
            ->where('admin_id', $adminId)
            ->pluck('id');

        // If no media items are complete, return early to prevent errors
        if ($completeMediaIds->isEmpty()) {
            return response()->json([
                'labels' => ['On time', 'Delayed'],
                'datasets' => [[
                    'data' => [0, 0],
                    'backgroundColor' => ['#A3E635', '#FFD700'], // Example colors
                ]],
            ]);
        }

        // Step 2: Determine which of these 'complete' media items were ever 'overdue'.
        // We'll query the 'statuses' table directly to check their history.
        $delayedCompleteMediaIds = DB::table('statuses')
            ->select('model_id')
            ->where('model_type', Media::class)
            ->whereIn('model_id', $completeMediaIds) // Only check statuses for currently complete media
            ->where('name', 'overdue') // Look for 'overdue' status in their history
            ->groupBy('model_id') // Get unique media IDs that were ever overdue
            ->pluck('model_id');

        // Step 3: Calculate counts
        $totalComplete = $completeMediaIds->count();
        $delayedCount = $delayedCompleteMediaIds->count();
        $onTimeCount = $totalComplete - $delayedCount;

        // Step 4: Prepare data in the desired format
        $data = [
            'labels' => ['On time', 'Delayed'],
            'datasets' => [[
                'data' => [$onTimeCount, $delayedCount],
                // These colors should ideally match your Angular component's theme
                'backgroundColor' => ['#A3E635', '#FFD700'], // Green for On time, Yellow/Orange for Delayed
            ]],
        ];

        return response()->json($data);
    }

    public function activeMediaLocations(Request $request)
    {
        Media::$withoutAppends = true;
        request()->merge(['active' => 1]);
        $media = Media::filter($request->all(), MediaFilter::class)->latest();

        return response()->json($media->get());
    }

    public function reports()
    {
        $reports = Report::where('reportable_type', Media::class)->orderByDesc('created_at')->paginate();
        return ReportedMediaAdminResource::collection($reports);
    }

    public function deleteReport(Report $report)
    {
        $report->delete();
        return response()->json([], 200);
    }
}
