<?php

namespace Vtlabs\Media\Filters;

use Vtlabs\Media\Models\Media;
use EloquentFilter\ModelFilter;
use Vtlabs\Report\Models\Block;
use Vtlabs\Report\Models\Report;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Media\Helpers\MediaHelper;
use Vtlabs\Media\Models\UserAuthorPreference;

class MediaFilter extends ModelFilter
{
    public function setup()
    {
        // $this->onlyShowPublished();
        //$this->onlyCategorizable();

        // sorting
        if (!$this->input('trending') && !$this->input('recent') && (!$this->input('_sort') && !$this->input('_order'))) {
            // default order
            $this->latest();
        }

        if ($this->input('_sort') && $this->input('_order')) {
            $sortColumn = $this->input('_sort');
            $sortOrder = $this->input('_order');
            if(in_array($sortColumn, ['id', 'created_at'])) {
                $this->orderBy($sortColumn, $sortOrder);
            } else {
                $this->latest();
            }
        }

        if (!MediaHelper::isAdmin() && !$this->input('assignee')) {
            Log::info('MediaFilter: setup called by user: ' . Auth::id());
            if (!$this->input('status')) {
                $this->otherCurrentStatus(['rejected', 'inreview']);
            }

            // public / non public posts handling
            Log::info('MediaFilter: user_id present: ' . !$this->input('user_id'));
            if (!$this->input('user')) {
                $this->where('is_public', true);
            }
        }

        if (!MediaHelper::isAdmin()) {
            if (!$this->input('status')) {
                $this->otherCurrentStatus(['rejected']);
            }
        }

        if (!MediaHelper::isAdmin() && $this->input('postal_code')) {
            $admin = UserService::postalcodeToAdmin($this->input('postal_code'));
            if($admin) {
                $this->where('admin_id', $admin->id);
            }
        }

        // filter out media posted by blocked users
        if (Auth::check() && !MediaHelper::isAdmin()) {
            // remove posts from blocked users
            $blockedUsers = Block::where('blocker_id', Auth::id())->where('blocker_type', User::class)
                ->where('blockable_id', Auth::id())->where('blockable_type', User::class)->get()->pluck(['blockable_id'])->toArray();
            $this->whereNotIn('user_id', $blockedUsers);

            // remove reported posts
            $reportedPosts = Report::where('reporter_id', Auth::id())->where('reporter_type', User::class)->where('reportable_type', Media::class)->get()->pluck(['reportable_id'])->toArray();
            $this->whereNotIn('id', $reportedPosts);
        }

        // filter for supervisors
        if(MediaHelper::isSupervisor()) {
            $this->whereIn('category_id', MediaHelper::supervisorCategoryIds());
        }
    }

    public function id($id)
    {
        return $this->where('id', $id);
    }

    public function search($search)
    {
        return $this->where(function ($query) use ($search) {
            return CoreHelper::searchTranslated($query, 'title', App::getLocale(), $search);
        });
    }

    public function title($title)
    {
        return CoreHelper::searchTranslated($this, 'title', App::getLocale(), $title);
    }

    public function recent($recent)
    {
        return $this->orderBy('created_at', 'desc');
    }

    public function isParent()
    {
        return $this->doesntHave('parent');
    }

    public function trending($trending)
    {
        return $this->orderBy('views_count', 'desc');
    }

    public function recommended($recommended)
    {
        if (Auth::id()) {
            $preferredAuthors = UserAuthorPreference::where('user_id', Auth::id())->get()->pluck(['author_id'])->toArray();
            return $this->whereHas('authors', function ($query) use ($preferredAuthors) {
                return $query->whereIn('media_authors.id', $preferredAuthors);
            });
        }
    }

    public function parent($parent)
    {
        return $this->where('parent_media_id', $parent);
    }

    public function category($id)
    {
        return $this->where(function ($query) use ($id) {
            $query->whereHas('category', function ($query) use ($id) {
                $query->where('id', $id)->orWhere('slug', $id);
            })
                ->orWherehas('subcategories', function ($query) use ($id) {
                    $query->where('id', $id)->orWhere('slug', $id);
                });
        });
    }

    public function genre($id)
    {
        return $this->whereHas('subcategories', function ($query) use ($id) {
            return $query->where('id', $id);
        });
    }

    public function mood($id)
    {
        return $this->whereHas('subcategories', function ($query) use ($id) {
            return $query->where('id', $id);
        });
    }

    public function author($id)
    {
        return $this->whereHas('authors', function ($query) use ($id) {
            return $query->where('authors.id', $id);
        });
    }

    // public function onlyShowPublished()
    // {
    //     $this->where('status', Media::STATUS_PUBLISHED);
    // }

    public function categorizable()
    {
        $this->whereNotNull('category_id');
    }

    public function following($following)
    {
        $following = Auth::user()->followings->pluck('id')->all();
        return $this->whereIn('user_id', $following)->orderBy('created_at', 'desc');
    }

    public function user($user)
    {
        $this->where('user_id', $user);
    }

    public function assignee($assignee)
    {
        $this->where('assignee_id', $assignee);
    }

    public function likedBy($likedBy)
    {
        return $this->where(function ($query) use ($likedBy) {
            $query->whereHas('likers', function ($query) use ($likedBy) {
                $query->where('user_id', $likedBy);
            });
        });
    }

    public function meta($meta)
    {
        foreach ($meta as $key => $value) {
            return $this->where('meta->' . $key, $value);
        }
    }

    public function status($status)
    {
        return $this->currentStatus($status);
    }

    public function active($active)
    {
        return $this->currentStatus(['open', 'reviewed', 'assigned', 'started', 'overdue']);
    }

    public function admin($admin)
    {
        return $this->where('admin_id', $admin);
    }
}
