<?php

namespace Vtlabs\Media\Models;

use EloquentFilter\Filterable;
use Vtlabs\Media\Models\Author;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Vtlabs\Media\Models\MediaOrder;
use Illuminate\Support\Facades\Auth;
use Rennokki\Rating\Traits\CanBeRated;
use Illuminate\Database\Eloquent\Model;
use Rennokki\Rating\Contracts\Rateable;
use Vtlabs\Report\Traits\CanBeReported;
use Spatie\Translatable\HasTranslations;
use Overtrue\LaravelLike\Traits\Likeable;
use Vtlabs\Core\Traits\CoreHasMediaTrait;
use Spatie\MediaLibrary\HasMedia\HasMedia;
use BeyondCode\Comments\Traits\HasComments;
use ChristianKuri\LaravelFavorite\Traits\Favoriteable;

class Media extends Model implements HasMedia, Rateable
{
    const STATUS_PUBLISHED = 'published';
    const STATUS_UNPUBLISHED = 'unpublished';
    const STATUS_DRAFT = 'draft';

    use Filterable,
        HasTranslations,
        Favoriteable,
        CoreHasMediaTrait,
        CanBeRated,
        Likeable,
        HasComments,
        CanBeReported;

    protected $table = 'media';

    protected $guarded = [];

    protected $with = ['category', 'subcategories', 'content'];

    protected $casts = [
        'title' => 'array',
        'description' => 'array',
        'short_description' => 'array',
        'meta' => 'array',
        'price' => 'float'
    ];

    protected $appends = ['favoritesCount', 'isFavourite', 'seasonCount', 'episode_count'];

    public $availableMediaConversions = [];

    public $translatable = ['title', 'description', 'short_description'];

    public static function saveHashtags($string) {
        preg_match_all('/#(\w+)/',$string, $matches);
        foreach($matches[0] as $match) {
            DB::table('media_hashtags')->updateOrInsert(['hashtag' => $match], []);
            DB::table('media_hashtags')->where('hashtag', $match)->increment('frequency');
        }
    }

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);  
        
        $this->availableMediaConversions = [
            'thumb' => ["width" => config('vtlabs_media.images.thumb', 50)],
            'small' => ["width" => config('vtlabs_media.images.small', 150)],
            'medium' => ["width" => config('vtlabs_media.images.medium', 300)],
            'large' => ["width" => config('vtlabs_media.images.large', 500)],
        ];
    }

    public function getIsFavouriteAttribute()
    {
        return $this->isFavorited();
    }

    public function getIsPurchasedAttribute()
    {
        if(Auth::id()) {
            $mediaOrder = MediaOrder::where('media_id', $this->id)->where('user_id', Auth::id())->latest()->first();
            $paymentStatus = $mediaOrder->payment->status ?? null;
            if($mediaOrder && $paymentStatus == 'paid') {
                return true;
            }
        }
        return false;
    }

    public function getSeasonCountAttribute()
    {
        return $this->episodes()->select('id')->groupBy('season_number')->count();
    }

    public function getEpisodeCountAttribute()
    {
        return $this->episodes()->count();
    }

    public function category()
    {
        return $this->belongsTo(config('vtlabs_category.models.category'), 'category_id');
    }

    public function subcategories()
    {
        return $this->belongsToMany(config('vtlabs_category.models.category'), 'media_categories');
    }

    public function content()
    {
        return $this->hasMany(config('vtlabs_media.models.media_content'), 'media_id');
    }

    public function episodes()
    {
        return $this->hasMany(config('vtlabs_media.models.media_episode'), 'parent_media_id');
    }

    public function authors()
    {
        return $this->belongsToMany(Author::class, 'media_authors');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function parent()
    {
        return $this->belongsTo(config('vtlabs_media.models.media'), 'parent_media_id');
    }
}
