<?php

namespace Vtlabs\Media\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Media\Models\Author;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Media\Filters\AuthorFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Models\UserAuthorPreference;
use Vtlabs\Media\Filters\UserAuthorPreferenceFilter;
use Vtlabs\Media\Http\Resources\Media\AuthorResource;
use Vtlabs\Media\Http\Resources\Media\AuthorRatingResource;
use Vtlabs\Media\Http\Resources\Media\UserAuthorPreferenceResource;

/**
 * @group  Author
 *
 * APIs for author
 */
class AuthorController extends Controller
{
    public function __construct()
    {
        // since this url can be used with or without Authorization, we are applying auth middleware,
        // if Authorization header was present, so that we can get current user using Auth::user()
        if (array_key_exists('HTTP_AUTHORIZATION', $_SERVER)) {
            $this->middleware('auth:api');
        }
    }

    public function index(Request $request)
    {
        $request->validate([]);

        $authors = Author::filter($request->all(), AuthorFilter::class);

        return AuthorResource::collection($authors->paginate());
    }

    /**
     * Author by ID
     * 
     * @responseFile responses/media/author/author.show.get.200.json
     */
    public function show(Author $author)
    {
        return new AuthorResource($author);
    }

    /**
     * Favourite Author list
     * 
     * @authenticated
     * 
     * @responseFile responses/media/author/author.list.get.200.json
     */
    public function favourites(Request $request)
    {
        return AuthorResource::collection(Auth::user()->favorite(Author::class)->flatten());
    }

    /**
     * Toggle Favourite Author
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleFavourite(Author $author, Request $request)
    {
        $author->toggleFavorite();

        return response()->json((object)[]);
    }

    /**
     * Rating Author list
     * 
     * @responseFile responses/media/author/rating.list.get.200.json
     */
    public function ratingList(Author $author, Request $request)
    {
        return AuthorRatingResource::collection($author->raters(Author::class)->paginate());
    }

    /**
     * Rate author
     * 
     * @authenticated
     * 
     * @bodyParam  rating float required Rating in number
     * @bodyParam  review string required Rating in text
     * 
     * @response []
     */
    public function ratingStore(Author $author, Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric',
            'review' => 'required',
            'created_at' => Carbon::now()
        ]);

        $user = Auth::user();

        $user->unrate($author);
        $user->rate($author, $request->rating, $request->review);

        return response()->json([], 200);
    }

    public function storeAuthorPreferences(Request $request)
    {
        $request->validate([
            'authors' => 'sometimes|array|exists:authors,id'
        ]);

        if ($request->authors) {
            if (count($request->authors) > 1) {
                // if api client is sending the array of authors we will delete all existing preferred authors and save new authors
                UserAuthorPreference::where('user_id', Auth::id())->delete();
                foreach ($request->authors as $authorId) {
                    UserAuthorPreference::create([
                        'author_id' => $authorId,
                        'user_id' => Auth::id()
                    ]);
                }
            }

            if (count($request->authors) == 1) {
                $authorId = $request->authors[0];
                $authorPreference = UserAuthorPreference::where('author_id', $authorId)->where('user_id', Auth::id())->exists();
                if ($authorPreference) {
                    // unfollow
                    UserAuthorPreference::where('author_id', $authorId)->where('user_id', Auth::id())->delete();
                } else {
                    // follow
                    UserAuthorPreference::create(['author_id' => $authorId, 'user_id' => Auth::id()]);
                }
            }
        }

        return response()->json((object)[], 200);
    }

    public function listAuthorPreferences(Request $request)
    {
        $request->merge([
            'user' => Auth::id()
        ]);

        $userAuthorPreferences = UserAuthorPreference::filter($request->all(), UserAuthorPreferenceFilter::class);

        return UserAuthorPreferenceResource::collection($userAuthorPreferences->paginate());
    }
}
