<?php

namespace Vtlabs\Media\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Media\Models\Media;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Media\Filters\MediaFilter;
use Vtlabs\Media\Models\MediaContent;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Http\Resources\Media\Admin\MediaAdminResource;

class MediaController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'is_parent' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $medias = Media::filter($request->all(), MediaFilter::class);

        return MediaAdminResource::collection($medias->paginate());
    }

    public function show($id)
    {
        return new MediaAdminResource(Media::findOrFail($id));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'sometimes|translation',
            'description_translations' => 'sometimes|translation',
            'short_description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'price' => 'sometimes|numeric',
            'length' => 'sometimes|string',
            'language' => 'sometimes|string',
            'artists' => 'sometimes|string',
            'release_date' => 'sometimes|string',
            'maturity_rating' => 'sometimes|string',
            'status' => 'required|string',
            'user_id' => 'sometimes|exists:users,id',
            'category_id' => 'sometimes|exists:categories,id',
            'subcategories' => 'sometimes|array|exists:categories,id',
            'authors' => 'sometimes|array|exists:authors,id',
            'images.*' => 'sometimes|image',
            'content.*.type' => 'sometimes|in:main,clip',
            'content.*.original_source' => 'sometimes',
            'content.*.title' => 'sometimes',
            'content.*.length' => 'sometimes',
            'content.*.image' => 'sometimes|image'
        ]);

        request()->merge([
            "title" => request()->input('_title') ?? null,
            "description" => request()->input('_description') ?? null,
            "short_description" => request()->input('_short_description') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $media = Media::create($request->only([
            'title', 'description', 'short_description', 'meta', 'price', 'length', 'language', 'artists',
            'release_date', 'maturity_rating', 'category_id', 'user_id'
        ]));

        $media->save();

        // authors
        if (!empty($request->authors)) {
            $media->authors()->sync($request->authors);
        }

        // subcategories
        if (!empty($request->subcategories)) {
            $media->subcategories()->sync($request->subcategories);
        }

        // images
        if (!empty($request->images)) {
            foreach ($request->images as $image) {
                $media->addMedia($image)->toMediaCollection("images");
            }
        }

        // content
        if (!empty($request->content)) {
            foreach ($request->content as $content) {
                $meta = [];

                if (!empty($content['title'])) {
                    $meta['title'] = $content['title'];
                }

                if (!empty($content['length'])) {
                    $meta['length'] = $content['length'];
                }

                $mediaContent = MediaContent::create([
                    'original_source' => $content['original_source'],
                    'type' => $content['type'] ?? 'main',
                    'meta' => !empty($meta) ? $meta : null,
                    'source' => '',
                    'media_id' => $media->id
                ]);
                if (!empty($content['image'])) {
                    $mediaContent->addMedia($content['image'])->toMediaCollection("images");
                }
            }
        }

        return new MediaAdminResource($media->fresh());
    }

    public function update($id, Request $request)
    {
        $media = Media::findOrFail($id);

        $request->validate([
            'title_translations' => 'required|translation',
            'description_translations' => 'sometimes|translation',
            'short_description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'price' => 'sometimes|numeric',
            'length' => 'sometimes|string',
            'language' => 'sometimes|string',
            'artists' => 'sometimes|string',
            'release_date' => 'sometimes|string',
            'maturity_rating' => 'sometimes|string',
            'status' => 'required|string',
            'user_id' => 'sometimes|exists:users,id',
            'category_id' => 'sometimes|exists:categories,id',
            'subcategories' => 'sometimes|array|exists:categories,id',
            'authors' => 'sometimes|array|exists:authors,id',
            'images.*' => 'sometimes|image',
            'content.*.id' => 'sometimes',
            'content.*.type' => 'sometimes|in:main,clip',
            'content.*.original_source' => 'sometimes',
            'content.*.title' => 'sometimes',
            'content.*.length' => 'sometimes',
            'content.*.image' => 'sometimes|image'
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "description" => request()->input('_description') ?? null,
            "short_description" => request()->input('_short_description') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $media->fill($request->only([
            'title', 'description', 'short_description', 'meta', 'price', 'length', 'language', 'artists',
            'release_date', 'maturity_rating', 'category_id', 'user_id'
        ]));

        $media->save();

        // subcategories
        if (!empty($request->subcategories)) {
            $media->subcategories()->sync($request->subcategories);
        }

        // authors
        if (!empty($request->authors)) {
            $media->authors()->sync($request->authors);
        }

        // images
        if (!empty($request->images)) {
            $media->clearMediaCollection('images');
            foreach ($request->images as $image) {
                $media->addMedia($image)->toMediaCollection("images");
            }
        }

        // content
        if (!empty($request->content)) {
            // $media->content()->delete(); // delete existing content
            foreach ($request->content as $content) {
                if (!empty($content['id'])) {
                    $mediaContent = MediaContent::find($content['id']);
                } else {
                    $mediaContent = new MediaContent();
                }

                $meta = [];

                if (!empty($content['title'])) {
                    $meta['title'] = $content['title'];
                }

                if (!empty($content['length'])) {
                    $meta['length'] = $content['length'];
                }

                $mediaContent->fill([
                    'original_source' => $content['original_source'],
                    'type' => $content['type'] ?? 'main',
                    'meta' => !empty($meta) ? $meta : null,
                    'source' => '',
                    'media_id' => $media->id
                ]);

                $mediaContent->save();

                if (!empty($content['image'])) {
                    $mediaContent->clearMediaCollection('images');
                    $mediaContent->addMedia($content['image'])->toMediaCollection("images");
                }
            }
        } else {
            $media->content()->delete(); // delete existing content
        }

        return new MediaAdminResource($media->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        Media::findOrFail($id)->delete();

        return response()->json([], 200);
    }
}
