<?php

namespace Vtlabs\Media\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Media\Models\Media;
use Illuminate\Support\Facades\Gate;;
use Vtlabs\Media\Models\MediaContent;
use Vtlabs\Media\Models\MediaEpisode;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Filters\MediaEpisodeFilter;
use Vtlabs\Media\Http\Resources\Media\Admin\MediaEpisodeAdminResource;

class EpisodeController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'parent' => 'required|exists:media,id',
            'media' => 'sometimes',
            'season_number' => 'sometimes'
        ]);

        $episodes = MediaEpisode::filter($request->all(), MediaEpisodeFilter::class);

        return MediaEpisodeAdminResource::collection($episodes->paginate());
    }

    public function show($id)
    {
        return new MediaEpisodeAdminResource(MediaEpisode::findOrFail($id));
    }

    public function store(Request $request)
    {
        $request->validate([
            'season_number' => 'required|numeric',
            'episode_number' => 'required|numeric',
            'parent_media_id' => 'required|exists:media,id',
            'title_translations' => 'required|translation',
            'description_translations' => 'required|translation',
            'short_description_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'category_id' => 'sometimes|exists:categories,id',
            'subcategories' => 'sometimes|array|exists:categories,id',
            'length' => 'required|string',
            'language' => 'required|string',
            'artists' => 'sometimes|string',
            'release_date' => 'required|string',
            'status' => 'required|string',
            'images.*' => 'sometimes|image',
            'content.*.type' => 'sometimes|in:main,clip',
            'content.*.original_source' => 'sometimes',
            'content.*.title' => 'sometimes'
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "description" => request()->input('_description') ?? null,
            "short_description" => request()->input('_short_description') ?? null,
            "maturity_rating" => '4+'
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $media = Media::create($request->only([
            'title', 'description', 'short_description', 'meta', 'price', 'length', 'language', 'artists',
            'release_date', 'maturity_rating', 'category_id', 'parent_media_id'
        ]));


        // subcategories
        if(!empty($request->subcategories)) {
            $media->subcategories()->sync($request->subcategories);
        }

        // images
        if (!empty($request->images)) {
            foreach ($request->images as $image) {
                $media->addMedia($image)->toMediaCollection("images");
            }
        }

        // content
        if(!empty($request->content)) {
            $mediaContent = [];
            foreach($request->content as $content) {
                $mediaContent[] = new MediaContent([
                    'original_source' => $content['original_source'],
                    'type' => $content['type'],
                    'meta' => ['title' => $content['title']],
                    'source' => ''
                ]);
            }
            $media->content()->saveMany($mediaContent);
        } else {
            $media->content()->delete(); // delete existing content
        }

        // create episode
        $episode = MediaEpisode::create([
            'season_number' => $request->season_number,
            'episode_number' => $request->episode_number,
            'media_id' => $media->id,
            'parent_media_id' => $request->parent_media_id
        ]);

        return new MediaEpisodeAdminResource($episode->fresh());
    }

    public function update($id, Request $request)
    {
        $episode = MediaEpisode::findOrFail($id);

        $request->validate([
            'season_number' => 'required|numeric',
            'episode_number' => 'required|numeric',
            'parent_media_id' => 'required|exists:media,id',
            'title_translations' => 'required|translation',
            'description_translations' => 'required|translation',
            'short_description_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'length' => 'required|string',
            'language' => 'required|string',
            'artists' => 'sometimes|string',
            'release_date' => 'required|string',
            'status' => 'required|string',
            'images.*' => 'sometimes|image',
            'content.*.type' => 'sometimes|in:main,clip',
            'content.*.original_source' => 'sometimes',
            'content.*.title' => 'sometimes'
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "description" => request()->input('_description') ?? null,
            "short_description" => request()->input('_short_description') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $episode->fill($request->only(['season_number', 'episode_number', 'parent_media_id']));

        $episode->media->fill($request->only([
            'title', 'description', 'short_description', 'meta', 'price', 'length', 'language', 'artists',
            'release_date', 'maturity_rating', 'category_id', 'parent_media_id'
        ]));

        $episode->media->save();

         // subcategories
         if(!empty($request->subcategories)) {
            $episode->media->subcategories()->sync($request->subcategories);
        }

        // images
        if (!empty($request->images)) {
            $episode->media->clearMediaCollection('images');
            foreach ($request->images as $image) {
                $episode->media->addMedia($image)->toMediaCollection("images");
            }
        }

        // content
        if(!empty($request->content)) {
            $mediaContent = [];
            foreach($request->content as $content) {
                $mediaContent[] = new MediaContent([
                    'original_source' => $content['original_source'],
                    'type' => $content['type'],
                    'meta' => ['title' => $content['title']],
                    'source' => '',
                ]);
            }
            $episode->media->content()->delete(); // delete existing content
            $episode->media->content()->saveMany($mediaContent);
        } else {
            $episode->media->content()->delete(); // delete existing content
        }

        return new MediaEpisodeAdminResource($episode->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        $episode = MediaEpisode::findOrFail($id);
        // also delete the corresponding media
        $episode->media->delete();

        $episode->delete();

        return response()->json([], 200);
    }
}
