<?php

namespace Vtlabs\Media\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Media\Models\Coupon;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Media\Filters\CouponFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Http\Resources\Media\Admin\CouponAdminResource;

class CouponController extends Controller
{
    public function index(Request $request)
    {
        $coupons = Coupon::filter($request->all(), couponFilter::class);

        return CouponAdminResource::collection($coupons->paginate());
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'detail_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'code' => 'required',
            'coins' => 'required|numeric',
            'expires_at' => 'required|date_format:Y-m-d',
            'image' => 'sometimes|image'
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "detail" => request()->input('_detail')
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $coupon = Coupon::create($request->only(['title', 'detail', 'meta', 'code', 'coins', 'expires_at']));

        if ($request->image) {
            $coupon->addMedia($request->image)->toMediaCollection("images");
        }

        return new CouponAdminResource($coupon->fresh());
    }

    public function show($id)
    {
        return new CouponAdminResource(Coupon::find($id));
    }

    public function update(Request $request, $id)
    {
        $coupon = Coupon::findOrFail($id);

        $request->validate([
            'title_translations' => 'required|translation',
            'detail_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'code' => 'required',
            'coins' => 'required|numeric',
            'expires_at' => 'required|date_format:Y-m-d',
            'image' => 'sometimes|image',
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "detail" => request()->input('_detail')
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $coupon->fill($request->only(['title', 'detail', 'meta', 'code', 'coins', 'expires_at']));
        $coupon->save();

        if ($request->image) {
            $coupon->clearMediaCollection('images');
            $coupon->addMedia($request->image)->toMediaCollection("images");
        }

        return new CouponAdminResource($coupon->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        Coupon::findOrFail($id)->delete();

        return response()->json([], 200);
    }
}
