<?php

namespace Vtlabs\Media\Http\Controllers\Api\Admin;

use ZipArchive;
use League\Csv\Reader;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Vtlabs\Media\Models\Author;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Media\Filters\AuthorFilter;
use Illuminate\Support\Facades\Storage;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Media\Http\Resources\Media\Admin\AuthorAdminResource;

class AuthorController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string',
            'scope' => 'sometimes|string',
            'parent' => 'sometimes|string'
        ]);

        $authors = Author::filter($request->all(), AuthorFilter::class);

        if ($request->pagination == '0') {
            $authors = $authors->get();
        } else {
            $authors = $authors->paginate();
        }
        return AuthorAdminResource::collection($authors);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|json|translation',
            'short_description_translations' => 'sometimes|json|translation',
            'description_translations' => 'sometimes|json|translation',
            'meta' => 'sometimes|json|nullable',
            'image' => 'sometimes|image'
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "short_description" => request()->input('_short_description') ?? null,
            "description" => request()->input('_description') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $author = Author::create($request->only(['title', 'short_description', 'description', 'meta']));

        if ($request->image) {
            $author->addMedia($request->image)->toMediaCollection("images");
        }

        return new AuthorAdminResource($author->fresh());
    }

    public function show($id)
    {
        return new AuthorAdminResource(Author::find($id));
    }

    public function update(Author $author, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|json|translation',
            'short_description_translations' => 'sometimes|json|translation',
            'description_translations' => 'sometimes|json|translation',
            'meta' => 'sometimes|json|nullable',
            'image' => 'sometimes|image'
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "short_description" => request()->input('_short_description') ?? null,
            "description" => request()->input('_description') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $author->fill($request->only(['title', 'short_description', 'description', 'meta']));
        $author->save();

        if ($request->image) {
            $author->clearMediaCollection("images");
            $author->addMedia($request->image)->toMediaCollection("images");
        }

        return new AuthorAdminResource($author->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        Author::findOrFail($id)->delete();

        return response()->json([], 200);
    }
}
