<?php

namespace Vtlabs\Media\Filters;

use Vtlabs\Media\Models\Media;
use EloquentFilter\ModelFilter;
use Vtlabs\Report\Models\Block;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Media\Models\UserAuthorPreference;

class MediaFilter extends ModelFilter
{
    public function setup()
    {
        $this->onlyShowPublished();
        //$this->onlyCategorizable();

        if (!$this->input('trending') && !$this->input('recent')) {
            $this->orderByDesc('created_at');
        }

        // filter out media posted by blocked users
        if (Auth::check()) {
            $blockedUsers = Block::where('blocker_id', Auth::id())->where('blocker_type', User::class)
                ->where('blockable_id', Auth::id())->where('blockable_type', User::class)->get()->pluck(['blockable_id'])->toArray();
            $this->whereNotIn('user_id', $blockedUsers);
        }
    }

    public function search($search)
    {
        return $this->where(function ($query) use ($search) {
            return CoreHelper::searchTranslated($query, 'title', App::getLocale(), $search)->orWhereHas('authors', function($query1) use($search) {
                return $query1->where(DB::raw('lower(title)'), 'LIKE', '%' . strtolower($search) . '%');
            });
        });
    }

    public function title($title)
    {
        return CoreHelper::searchTranslated($this, 'title', App::getLocale(), $title);
    }

    public function recent($recent)
    {
        return $this->orderBy('created_at', 'desc');
    }

    public function isParent()
    {
        return $this->doesntHave('parent');
    }

    public function trending($trending)
    {
        return $this->orderBy('views_count', 'desc');
    }

    public function recommended($recommended)
    {
        if (Auth::id()) {
            $preferredAuthors = UserAuthorPreference::where('user_id', Auth::id())->get()->pluck(['author_id'])->toArray();
            return $this->whereHas('authors', function ($query) use ($preferredAuthors) {
                return $query->whereIn('media_authors.id', $preferredAuthors);
            });
        }
    }

    public function parent($parent)
    {
        return $this->where('parent_media_id', $parent);
    }

    public function category($id)
    {
        return $this->where(function ($query) use ($id) {
            $query->whereHas('category', function ($query) use ($id) {
                $query->where('id', $id)->orWhere('slug', $id);
            })
                ->orWherehas('subcategories', function ($query) use ($id) {
                    $query->where('id', $id)->orWhere('slug', $id);
                });
        });
    }

    public function genre($id)
    {
        return $this->whereHas('subcategories', function ($query) use ($id) {
            return $query->where('id', $id);
        });
    }

    public function mood($id)
    {
        return $this->whereHas('subcategories', function ($query) use ($id) {
            return $query->where('id', $id);
        });
    }

    public function author($id)
    {
        return $this->whereHas('authors', function ($query) use ($id) {
            return $query->where('authors.id', $id);
        });
    }

    public function onlyShowPublished()
    {
        $this->where('status', Media::STATUS_PUBLISHED);
    }

    public function categorizable()
    {
        $this->whereNotNull('category_id');
    }

    public function following($following)
    {
        $following = Auth::user()->followings->pluck('id')->all();
        return $this->whereIn('user_id', $following)->orderBy('created_at', 'desc');
    }

    public function user($user)
    {
        $this->where('user_id', $user);
    }

    public function likedBy($likedBy)
    {
        return $this->where(function ($query) use ($likedBy) {
            $query->whereHas('likers', function ($query) use ($likedBy) {
                $query->where('user_id', $likedBy);
            });
        });
    }

    public function meta($meta)
    {
        foreach ($meta as $key => $value) {
            return $this->where('meta->' . $key, $value);
        }
    }
}
