<?php

namespace Vtlabs\Media\Models;

use Exception;
use Carbon\Carbon;
use EloquentFilter\Filterable;
use Vtlabs\Core\Models\User\User;
use Illuminate\Database\Eloquent\Model;
use Spatie\Translatable\HasTranslations;
use Vtlabs\Core\Traits\CoreHasMediaTrait;
use Spatie\MediaLibrary\HasMedia\HasMedia;

class Coupon extends Model implements HasMedia
{
    use Filterable,
        CoreHasMediaTrait,
        HasTranslations;
        
    protected $table = 'media_coupons';

    protected $guarded = [];

    protected $casts = [
        'title' => 'array',
        'detail' => 'array',
        'meta' => 'array',
        'coins' => 'integer'
    ];

    public $translatable = ['title', 'detail'];

    public function users()
    {
        return $this->belongsToMany(User::class, 'media_coupons_users');
    }

    /**
     * @throws Exception
     */
    public static function getCouponFromCode(string $couponCode, User $user): Coupon
    {
        try {
            $coupon = Coupon::where('code', $couponCode)->firstOrFail();
        } catch(\Exception $ex) {
            throw new Exception('Coupon not found');
        }

        $isUsed = $coupon->users()->wherePivot('user_id', $user->id)->exists();

        if($isUsed) {
            throw new \Exception('Coupon already used');
        }

        $isExpired = $coupon->expires_at ? Carbon::now()->gte($coupon->expires_at) : false;

        if($isExpired) {
            throw new \Exception('Coupon expired');
        }

        return $coupon;
    }
}
