<?php

namespace Vtlabs\Media\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Overtrue\LaravelLike\Like;
use Vtlabs\Media\Models\Media;
use Vtlabs\Media\Models\Comment;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Core\Models\Notification;
use Vtlabs\Media\Filters\MediaFilter;
use Vtlabs\Media\Models\MediaContent;
use Vtlabs\Media\Models\MediaEpisode;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Core\Http\Resources\UserResource;
use Vtlabs\Media\Filters\MediaEpisodeFilter;
use Vtlabs\Core\Helpers\PushNotificationHelper;
use Vtlabs\Media\Http\Resources\Media\MediaResource;
use Vtlabs\Media\Http\Resources\Media\CommentResource;
use Vtlabs\Media\Http\Resources\Media\MediaRatingResource;
use Vtlabs\Media\Http\Resources\Media\MediaEpisodeResource;
use League\Flysystem\Filesystem;
use League\Flysystem\AwsS3v3\AwsS3Adapter;
use Storage;
use Str;

/**
 * @group  Media
 *
 * APIs for media
 */
class MediaController extends Controller
{
    public function __construct()
    {
        // since this url can be used with or without Authorization, we are applying auth middleware,
        // if Authorization header was present, so that we can get current user using Auth::user()
        if (array_key_exists('HTTP_AUTHORIZATION', $_SERVER)) {
            $this->middleware('auth:api');
        }
    }

    /**
     * List of media
     *
     * @queryParam  search string Search media.
     * @queryParam  recent boolean Recent media items. Example: 1
     * @queryParam  category string Media by category id.
     * @queryParam  author string Media by author id.
     * 
     * @responseFile responses/media/media.list.get.200.json
     */
    public function index(Request $request)
    {
        $request->validate([
            'search' => 'sometimes',
            'recent' => 'sometimes|boolean',
            'category' => 'sometimes',
            'categorizable' => 'sometimes|boolean',
            'following' => 'sometimes|boolean',
            'author' => 'sometimes|exists:authors,id',
            'user' => 'sometimes|exists:users,id',
            'liked_by' => 'sometimes|exists:users,id',
            'is_parent' => 'sometimes|boolean'
        ]);

        $mediaItems = Media::filter($request->all(), MediaFilter::class);

        return MediaResource::collection($mediaItems->paginate());
    }

    /**
     * Media by ID
     * 
     * @responseFile responses/media/media.show.get.200.json
     */
    public function show(Media $media)
    {
        $media->save();
        return new MediaResource($media);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'sometimes',
            'meta' => 'sometimes|json|nullable',
            'category_id' => 'sometimes|exists:categories,id',
            'image_urls' => 'sometimes|array',
            'content' => "sometimes|nullable|array",
            'content.*.original_source' => 'required'
        ]);

        request()->merge([
            "user_id" => Auth::id()
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        // save hastags
        Media::saveHashtags($request->title);

        $media = Media::create($request->only([
            'title', 'meta', 'user_id', 'category_id'
        ]));

        $media->save();

        // images
        if (!empty($request->image_urls)) {
            foreach ($request->image_urls as $image) {
                $media->addMediaFromUrl($image)->toMediaCollection("images");
            }
        }

        // content
        if (!empty($request->content)) {
            $mediaContent = [];
            foreach ($request->content as $content) {
                $mediaContent[] = new MediaContent([
                    'original_source' => $content['original_source'],
                    'type' => 'main',
                    'source' => ''
                ]);
            }
            $media->content()->saveMany($mediaContent);
        }

        return new MediaResource($media->fresh());
    }

    public function uploadMedia(Media $media, Request $request)
    {
        $request->validate([
            'file' => 'required',
            'file_width' => 'required',
            'file_height' => 'required',
            'file_orientation' => 'required'
        ]);

        try {
            $s3 = Storage::disk('s3');
            $client = $s3->getDriver()->getAdapter()->getClient();
            $adapter = new AwsS3Adapter($client, \Config::get('filesystems.disks.s3.bucket'));
            $filesystem = new Filesystem($adapter);

            $file = $request->file('file');
            $filename = Str::uuid() . '.' .  $file->getClientOriginalExtension();

            $filesystem->writeStream(
                $filename,
                fopen($file->getRealPath(), 'r'),
                [
                    'Metadata' => [
                        'width' => $request->file_width,
                        'height' => $request->file_height,
                        'file_orientation' => $request->file_orientation,
                        'post_id' => $media->id
                    ],
                ]
            );

            // attach uploaded video with media
            $mediaContent[] = new MediaContent([
                'original_source' => 'https://' . 'neplay' . '.s3.' . \Config::get('filesystems.disks.s3.region') . '.amazonaws.com/' . explode('.', $filename)[0] . '/' . $filename,
                'type' => 'main',
                'source' => ''
            ]);

            $media->content()->saveMany($mediaContent);
        } catch (\Exception $ex) {
            // handle file upload on S3
        }

        return new MediaResource($media->fresh());
    }

    public function updateTranscodingComplete(Media $media, Request $request)
    {
        $request->validate([
            'complete' => 'sometimes|numeric',
            'count' => 'sometimes|numeric'
        ]);

        if ($request->complete) {
            $meta = $media->meta;
            $meta['transcoding'] = 1;
            $media->meta = $meta;
            $media->save();
        }

        if ($request->count) {
            $meta = $media->meta;
            $meta['transcoding_count'] = $request->count;
            $media->meta = $meta;
            $media->save();
        }

        return response()->json([], 200);
    }

    public function updateCount(Media $media, Request $request)
    {
        $request->validate([
            'shares_count' => 'sometimes|boolean',
            'views_count' => 'sometimes|boolean',
        ]);

        if ($request->shares_count) {
            $media->shares_count++;
            $media->save();
        }

        if ($request->views_count) {
            // when user view the video update his coins balance
            if (Auth::user() && $media['meta']['type'] != 'music') {
                $settings = CoreHelper::settingsAsDictionary();
                $coinsOnView = isset($settings['coins_on_view']) ? $settings['coins_on_view'] : 0;
                if ($coinsOnView) {
                    // do not credit coins if already watched
                    if (Auth::user()->transactions()->where('meta->media_id', $media->id)->count() == 0) {
                        Auth::user()->deposit(intVal($coinsOnView), 'deposit', ['media_id' => $media->id]);
                    }
                }
            }

            $media->views_count++;
            $media->save();
        }

        return response()->json((object)[]);
    }

    public function destroy(Media $media)
    {
        $media->delete();

        return response()->json((object)[], 200);
    }

    /**
     * Favourite Media list
     * 
     * @authenticated
     * 
     * @responseFile responses/media/media.list.get.200.json
     */
    public function favourites(Request $request)
    {
        $request->validate([
            'category' => 'sometimes|nullable'
        ]);

        // apply category filter

        $mediaList = Auth::user()->favorites()->with('favoriteable')->where('favoriteable_type', Media::class)->orderByDesc('created_at')->get()->pluck(['favoriteable']);

        // remove null values
        $mediaList = $mediaList->filter();

        if ($request->category) {
            $categoryId = $request->category;
            $mediaList = $mediaList->filter(function ($media, $key) use ($categoryId) {
                // compare id or slug of the category
                return $media->category->id == $categoryId || $media->category->slug == $categoryId;
            });
        }

        return MediaResource::collection($mediaList);
    }

    /**
     * Toggle Favourite
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleFavourite(Media $media, Request $request)
    {
        $media->toggleFavorite();

        return response()->json((object)[]);
    }

    public function favouriteMediaImages()
    {
        // api to get images of favourite media, used to show media images in UI sections
        $favMedias = Auth::user()->favorites()->where('favoriteable_type', Media::class)->orderByDesc('created_at')->limit(4)->get()->pluck(['favoriteable_id'])->toArray();
        $medias = Media::whereIn('id', $favMedias)->get();

        $images = [];

        foreach ($medias as $media) {
            array_push($images, $media->getMediaUrlsAttribute()["images"][0]["default"]);
        }

        return response()->json(['media_images' => $images]);
    }

    /**
     * List of episodes
     *
     * @queryParam  season int Season number
     * 
     * @responseFile responses/media/episode.list.get.200.json
     */
    public function episodes(Media $media, Request $request)
    {
        $request->validate([
            'season' => 'sometimes|number'
        ]);

        $episodes = MediaEpisode::where('parent_media_id', $media->id)->filter($request->all(), MediaEpisodeFilter::class)->ordered();
        return MediaEpisodeResource::collection($episodes->get());
    }

    /**
     * Rating Media list
     * 
     * @responseFile responses/media/rating.list.get.200.json
     */
    public function ratingList(Media $media, Request $request)
    {
        return MediaRatingResource::collection($media->raters(User::class)->paginate());
    }

    /**
     * Rate media
     * 
     * @authenticated
     * 
     * @bodyParam  rating float required Rating in number
     * @bodyParam  review string required Rating in text
     * 
     * @response []
     */
    public function ratingStore(Media $media, Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric',
            'review' => 'required',
            'created_at' => Carbon::now()
        ]);

        $user = Auth::user();

        $user->unrate($media);
        $user->rate($media, $request->rating, $request->review);

        return response()->json((object)[], 200);
    }

    /**
     * Get rating summary
     * 
     * @responseFile responses/media/rating.summary.get.200.json
     */
    public function ratingSummary(Media $media)
    {
        return response()->json([
            "average_rating" => $media->averageRating(User::class),
            "total_ratings" => $media->raters(User::class)->count(),
            "summary" => DB::table('ratings')->selectRaw('count(*) as total, ROUND(rating) as rounded_rating')
                ->where('rateable_type', Media::class)
                ->where('rateable_id', $media->id)
                ->where('rater_type', User::class)
                ->groupBy('rounded_rating')
                ->get()
        ]);
    }

    /**
     * Toggle Like
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleLike(Media $media, Request $request)
    {
        Auth::user()->toggleLike($media);

        $liked = Auth::user()->hasLiked($media);

        if ($liked && $media->user_id != Auth::id()) {
            Notification::create([
                'type' => 'like',
                'text' => 'liked your video',
                'meta' => ['media' => new MediaResource($media)],
                'user_id' => $media->user_id,
                'from_user_id' => Auth::id()
            ]);

            $media->user->sendPushNotification(
                'customer',
                __('vtlabs_media::messages.notification_like_post_title', ['liked_by' => Auth::user()->name]),
                __('vtlabs_media::messages.notification_like_post_body'),
                ['media_id' => $media->id]
            );
        }

        return response()->json(["like" => $liked]);
    }

    /**
     * List of likers
     * 
     * @authenticated
     * 
     * @response []
     */
    public function likers(Media $media, Request $request)
    {
        return UserResource::collection($media->likers()->paginate());
    }

    /**
     * List of liked media
     * 
     * @authenticated
     * 
     * @response []
     */
    public function likes(Request $request)
    {
        $likes = Like::withType('Vtlabs\Media\Models\Media')->where('user_id', Auth::id())->get()->pluck(['likeable_id'])->toArray();

        return MediaResource::collection(Media::whereIn($likes)->paginate());
    }

    /**
     * List of comments
     * 
     * @authenticated
     * 
     * @response []
     */
    public function listComments(Media $media, Request $request)
    {
        return CommentResource::collection($media->comments()->paginate());
    }

    public function listReplyComments(Comment $comment, Request $request)
    {
        return CommentResource::collection($comment->comments()->paginate());
    }

    /**
     * Post a comment
     * 
     * @authenticated
     * 
     * @response []
     */
    public function createComment(Media $media, Request $request)
    {
        $request->validate([
            'comment' => 'required|string'
        ]);

        $comment = $media->comment($request->comment);

        if ($media->user_id != Auth::id()) {
            Notification::create([
                'type' => 'comment',
                'text' => 'commented on your video',
                'meta' => ['media' => new MediaResource($media)],
                'user_id' => $media->user_id,
                'from_user_id' => Auth::id()
            ]);

            $media->user->sendPushNotification(
                'customer',
                __('vtlabs_media::messages.notification_comment_post_title', ['comment_by' => Auth::user()->name]),
                __('vtlabs_media::messages.notification_comment_post_body'),
                ['media_id' => $media->id]
            );
        }

        return new CommentResource($comment);
    }

    /**
     * Post a comment reply
     * 
     * @authenticated
     * 
     * @response []
     */
    public function replyComment(Comment $comment, Request $request)
    {
        $request->validate([
            'comment' => 'required|string'
        ]);

        $newcomment = $comment->comment($request->comment);

        return new CommentResource($newcomment);
    }

    /**
     * Toggle comment Like
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleCommentLike(Comment $comment, Request $request)
    {
        Auth::user()->toggleLike($comment);

        $liked = Auth::user()->hasLiked($comment);

        if ($liked && $comment->user_id != Auth::id()) {
            Notification::create([
                'type' => 'comment_like',
                'text' => 'liked your comment',
                'meta' => ['comment' => new CommentResource($comment)],
                'user_id' => $comment->user_id,
                'from_user_id' => Auth::id()
            ]);

            $media = Media::find($comment->commentable_id);
            $media->user->sendPushNotification(
                'customer',
                __('vtlabs_media::messages.notification_like_comment_title', ['liked_by' => Auth::user()->name]),
                __('vtlabs_media::messages.notification_like_comment_body'),
                ['media_id' => $media->id]
            );
        }

        return response()->json(["like" => $liked]);
    }

    /**
     * List of likers
     * 
     * @authenticated
     * 
     * @response []
     */
    public function commentLikers(Comment $comment, Request $request)
    {
        return UserResource::collection($comment->likers()->paginate());
    }

    public function updateComment(Comment $comment, Request $request)
    {
        $request->validate([
            'comment' => 'required|string'
        ]);

        $comment->comment = $request->comment;

        $comment->save();

        return new CommentResource($comment);
    }

    public function deleteComment(Comment $comment, Request $request)
    {
        $comment->delete();

        return response()->json((object)[], 200);
    }

    public function stats(User $user)
    {
        $medias = Media::where('user_id', $user->id)->get()->pluck(['id'])->toArray();

        return response()->json([
            'total_likes' => Like::withType('Vtlabs\Media\Models\Media')->whereIn('likeable_id', $medias)->count()
        ]);
    }

    public function repost(Media $media, Request $request)
    {
        $user = Auth::user();

        $repostMedia = $media->replicate()->fill([
            'description' => null,
            'short_description' => null,
            'views_count' => 0,
            'shares_count' => 0,
            'user_id' => $user->id,
            'meta' => array_merge($media->meta ?? [], ['original' => new MediaResource($media)]),
            'created_at' => now(),
            'updated_at' => now()
        ]);

        $repostMedia->save();

        Notification::create([
            'type' => 'repost',
            'text' => 'reposted your post',
            'meta' => [],
            'user_id' => $media->user_id,
            'from_user_id' => Auth::id()
        ]);

        $media->user->sendPushNotification(
            'customer',
            __('vtlabs_media::messages.notification_repost_title', ['repost_by' => Auth::user()->name]),
            __('vtlabs_media::messages.notification_repost_body'),
            ['media_id' => $media->id]
        );

        return new MediaResource($repostMedia->refresh());
    }

    public function report(Media $media, Request $request)
    {
        $request->validate([
            'reason' => 'sometimes'
        ]);

        $user = User::find(Auth::id());

        $media->report($user, ['reason' => $request->reason]);

        return response([], 200);
    }

    public function getPopularHashtags(Request $request)
    {
        $request->validate([
            'limit' => 'sometimes|numeric'
        ]);

        $limit = $request->limit ?? 10;

        return response()->json(DB::table('media_hashtags')->orderByDesc('frequency')->limit($limit)->get());
    }

    public function notifyFollowers()
    {
        $user = Auth::user();

        $notifyIds = $user->followers->pluck(['notification'])->flatten()->filter()->toArray();

        $oneSignal = PushNotificationHelper::getOneSignalInstance('customer');

        $data['title'] = __('vtlabs_media::messages.notification_live_title', ['name' => $user->name]);
        $data['body'] = __('vtlabs_media::messages.notification_live_body', ['name' => $user->name]);

        $oneSignal->sendNotificationToUser(
            $data['title'],
            $notifyIds,
            null,
            array_merge($data, ['user_id' => $user->id])
        );

        return response()->json((object)[]);
    }

    public function uploadVideo(Media $media, Request $request)
    {
        $request->validate([
            'file' => 'required|file'
        ]);

        try {
            $s3 = Storage::disk('s3');
            $client = $s3->getDriver()->getAdapter()->getClient();
            $adapter = new AwsS3Adapter($client, \Config::get('filesystems.disks.s3.bucket'));
            $filesystem = new Filesystem($adapter);

            $file = $request->file('file');
            $filename = Str::uuid() . '.' .  $file->getClientOriginalExtension();

            $filesystem->writeStream(
                $filename,
                fopen($file->getRealPath(), 'r'),
                [
                    'Metadata' => [
                        'width' => 100,
                        'height' => 100,
                    ],
                ]
            );

            // attach uploaded video with media

            $mediaContent[] = new MediaContent([
                'original_source' => $content['original_source'],
                'type' => 'main',
                'source' => ''
            ]);

            $media->content()->saveMany($mediaContent);
            return new MediaResource($media->fresh());
        } catch (\Exception $ex) {
            return response()->json(['message' => $ex->getMessage()], 400);
        }
    }
}
