<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Flat;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Gateapp\Models\Visitorlog;
use Vtlabs\Gateapp\Events\VisitorlogEvent;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Filters\VisitorlogFilter;
use Vtlabs\Gateapp\Http\Resources\VisitorlogResource;

class VisitorlogController extends Controller
{
    public function index(Request $request)
    {
        $visitorlogs = Visitorlog::filter($request->all(), VisitorlogFilter::class)->latest()->paginate();

        return VisitorlogResource::collection($visitorlogs);
    }

    public function show(Visitorlog $visitorlog)
    {
        return new VisitorlogResource($visitorlog);
    }
    
    public function store(Request $request)
    {
        $request->validate([
            'type' => 'required',
            'name' => 'sometimes|nullable',
            'flat_id' => 'required|exists:gateapp_flats,id',
            'contact' => 'sometimes|nullable',
            'company_name' => 'sometimes|nullable',
            'vehicle_number' => 'sometimes|nullable',
            'pax' => 'sometimes|nullable|integer',
            'intime' => 'sometimes|nullable|date_format:Y-m-d H:i:s',
            'meta' => 'sometimes|nullable|json',
            'status' => 'sometimes'
        ]);

        $flat = Flat::find($request->flat_id);
        
        $request->merge([
            'building_id' => $flat->building_id,
            'project_id' => $flat->building->project_id,
            'meta' => $request->meta ? json_decode($request->meta, true) : null,
            'user_id' => Auth::id()
        ]);

        $visitorlog = Visitorlog::create($request->only(['type', 'name', 'contact', 'company_name', 'vehicle_number', 'pax' ,'intime', 'flat_id', 'building_id' ,'project_id', 'user_id', 'meta']));

        if($request->status == 'preapproved') {
            $oldStatus = $visitorlog->status;
            $visitorlog->setStatus('preapproved');
            $visitorlog->save();
            event(new VisitorlogEvent($visitorlog, false, $oldStatus));
        } else {
            event(new VisitorlogEvent($visitorlog, true));
        }

        return new VisitorlogResource($visitorlog);
    }

    public function update(Visitorlog $visitorlog, Request $request)
    {
        $request->validate([
            'intime' => 'sometimes|nullable|date_format:Y-m-d H:i:s',
            'outtime' => 'sometimes|nullable|date_format:Y-m-d H:i:s',
            'status' => 'required',
            'type' => 'sometimes'
        ]);

        $oldStatus = $visitorlog->status;

        if(Str::contains($visitorlog->type, 'household_')){
            // create a new record of vistorlog for log purpose of households
            $visitorlog = $visitorlog->replicate()->fill([
                'type' => $request->type,
                'intime' => $request->intime,
                'user_id' => Auth::id()
            ]);
            $visitorlog->save();
            $visitorlog->setStatus($request->status);
            $visitorlog->save();
        } else {
            $visitorlog->setStatus($request->status);
            $visitorlog->fill($request->only(['outtime', 'intime']));
            $visitorlog->save();
        }

        event(new VisitorlogEvent($visitorlog, false, $oldStatus));

        return new VisitorlogResource($visitorlog);
    }

    public function destroy(Visitorlog $visitorlog)
    {
        Gate::authorize('delete');

        $visitorlog->delete();

        return response()->json([], 204);
    }
}
