<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Vtlabs\Gateapp\Models\Amenity;
use Vtlabs\Gateapp\Models\Resident;
use Vtlabs\Appointment\Models\Appointment;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Filters\AppointmentFilter;
use Vtlabs\Appointment\Services\AppointmentService;
use Vtlabs\Gateapp\Http\Resources\AppointmentResource;

class AppointmentController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'resident_id' => 'required'
        ]);

        $appointments = Appointment::filter($request->all(), AppointmentFilter::class)->orderBy('date', 'desc');

        return AppointmentResource::collection($appointments->paginate());
    }

    public function store(Amenity $amenity, Request $request)
    {
        $request->validate([
            'resident_id' => 'required'
        ]);

        $resident = Resident::find($request->resident_id);

        $request->merge(['time_to' => Carbon::createFromFormat('H:i',$request->time_to)->subMinute()->format('H:i')]);

        $request->validate(AppointmentService::validationRules());
        
        if($amenity->isSlotOccupied($request->date, $request->date_to, $request->time_from, $request->time_to)) {
            return response()->json(['message' => 'Slot already occupied'], 400);
        }

        $appointment = $resident->appoint($amenity, $request->all());

        return new AppointmentResource($appointment);
    }

    public function update(Appointment $appointment, Request $request)
    {
        $request->validate([
            'status' => ['required', Rule::in(config('vtlabs_appointment.appointment.status_list'))]
        ]);

        AppointmentService::updateStatus($appointment, $request->status);

        return new AppointmentResource($appointment->fresh());
    }
}
