<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api\Admin;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Guard;
use Vtlabs\Gateapp\Models\Project;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Gateapp\Filters\GuardFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Http\Resources\Admin\GuardAdminResource;

class GuardController extends Controller
{
    public function index(Request $request)
    {
        $guards = Guard::filter($request->all(), GuardFilter::class)->paginate();

        $projectId = Project::permissionedProject();
        if($projectId) {
            $guards = $guards->where('project_id', $projectId);
        }

        return GuardAdminResource::collection($guards);
    }

    public function show(Guard $guard)
    {
        return new GuardAdminResource($guard);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'mobile_number' => 'required|unique:users',
            'project_id' => 'required|exists:gateapp_projects,id',
            'meta' => 'sometimes|json|nullable',
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta, true) : null]);
        }

        // create user for guard first  
        $request->merge(['password' => bcrypt(Str::random(32))]);
        $userData = collect($request->only(['name', 'mobile_number', 'password']));
        $userData = $userData->merge(['mobile_verified' => 1, 'roles' => ['guard']]);
        $user = UserService::create($userData);

        request()->merge(['user_id' => $user->id]);

        $guard = Guard::create($request->only(['project_id', 'user_id']));

        return new GuardAdminResource($guard);
    }

    public function update(Request $request, Guard $guard)
    {
        $request->validate([
            'project_id' => 'required|exists:gateapp_projects,id',
            'meta' => 'sometimes|json|nullable',
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta, true) : null]);
        }

        $guard = Guard::create($request->only(['project_id']));
        $guard->save();

        return new GuardAdminResource($guard);
    }

    public function destroy(Guard $guard)
    {
        Gate::authorize('delete');

        $guard->delete();

        return response()->json([], 204);
    }
}
