<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Project;
use Vtlabs\Gateapp\Models\Building;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Gateapp\Filters\BuildingFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Http\Resources\Admin\BuildingAdminResource;

class BuildingController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $buildings = Building::filter($request->all(), BuildingFilter::class);

        $projectId = Project::permissionedProject();
        if($projectId) {
            $buildings = $buildings->where('project_id', $projectId);
        }

        if ($request->pagination == '0') {
            $buildings = $buildings->get();
        } else {
            $buildings = $buildings->paginate();
        }

        return BuildingAdminResource::collection($buildings);
    }

    public function show(Building $building)
    {
        return new BuildingAdminResource($building);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'meta' => 'sometimes|json|nullable',
            'project_id' => 'required|exists:gateapp_projects,id',
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta, true) : null]);
        }

        $building = Building::create($request->all());

        return new BuildingAdminResource($building);
    }

    public function update(Request $request, Building $building)
    {
        $request->validate([
            'title' => 'required',
            'meta' => 'sometimes|json|nullable',
            'project_id' => 'required|exists:gateapp_projects,id',
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta, true) : null]);
        }
        
        $building->fill($request->all());
        $building->save();

        return new BuildingAdminResource($building);
    }

    public function destroy(Building $building)
    {
        Gate::authorize('delete');
        
        $building->delete();

        return response()->json([], 204);
    }
}
