<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Project;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Gateapp\Models\Announcement;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Models\AnnouncementOption;
use Vtlabs\Gateapp\Filters\AnnouncementFilter;
use Vtlabs\Gateapp\Http\Resources\Admin\AnnouncementAdminResource;

class AnnouncementController extends Controller
{
    public function index(Request $request)
    {
        $announcements = Announcement::filter($request->all(), AnnouncementFilter::class);

        $projectId = Project::permissionedProject();
        if($projectId) {
            $announcements = $announcements->where('project_id', $projectId);
        }

        return AnnouncementAdminResource::collection($announcements->paginate());
    }

    public function show(Announcement $announcement)
    {
        return new AnnouncementAdminResource($announcement);
    }

    public function store(Request $request)
    {
        $request->validate([
            'type' => 'required',
            'message' => 'required',
            'posted_by' => 'required',
            'duedate' => 'sometimes|date_format:Y-m-d',
            'meta' => 'sometimes|json|nullable',
            'project_id' => 'required|exists:gateapp_projects,id',
            'options' => 'sometimes|array',
            'options.*.title' => 'required'
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta, true) : null]);
        }

        $announcement = Announcement::create($request->only(['type', 'message', 'posted_by', 'duedate', 'meta', 'project_id']));

        if ($request->options) {
            foreach ($request->options as $option) {
                AnnouncementOption::create([
                    'announcement_id' => $announcement->id,
                    'title' => $option['title']
                ]);
            }
        }

        return new AnnouncementAdminResource($announcement);
    }

    public function update(Request $request, Announcement $announcement)
    {
        $request->validate([
            'type' => 'required',
            'message' => 'required',
            'posted_by' => 'required',
            'duedate' => 'sometimes|date_format:Y-m-d',
            'meta' => 'sometimes|json|nullable',
            'project_id' => 'required|exists:gateapp_projects,id',
            'options' => 'sometimes|array',
            'options.*.title' => 'required'
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta, true) : null]);
        }

        $announcement->fill($request->only(['type', 'message', 'posted_by', 'duedate', 'meta', 'project_id']));
        $announcement->save();

        if ($request->options) {
            $announcement->options()->delete();
            foreach ($request->options as $option) {
                AnnouncementOption::create([
                    'announcement_id' => $announcement->id,
                    'title' => $option['title']
                ]);
            }
        }

        return new AnnouncementAdminResource($announcement);
    }

    public function destroy(Announcement $announcement)
    {
        Gate::authorize('delete');

        $announcement->delete();

        return response()->json([], 204);
    }
}
