<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api;

use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Flat;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Gateapp\Models\Complaint;
use Vtlabs\Gateapp\Filters\ComplaintFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Http\Resources\ComplaintResource;

class ComplaintController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'project_id' => 'sometimes',
            'flat_id' => 'sometimes',
            'status' => 'sometimes'
        ]);

        $complaints = Complaint::filter($request->all(), ComplaintFilter::class)->latest()->paginate();

        return ComplaintResource::collection($complaints);
    }

    public function show(Complaint $complaint)
    {
        return new ComplaintResource($complaint);
    }

    public function store(Request $request)
    {
        $request->validate([
            'type' => 'required',
            'message' => 'required',
            'meta' => 'sometimes|nullable|json',
            'flat_id' => 'required|exists:gateapp_flats,id'
        ]);

        $flat = Flat::find($request->flat_id);

        $request->merge([
            'project_id' => $flat->building->project_id,
            'meta' => $request->meta ? json_decode($request->meta) : null,
            'user_id' => Auth::id()
        ]);

        $complaint = Complaint::create($request->only(['type', 'message', 'meta', 'flat_id', 'project_id', 'user_id']));

        return new ComplaintResource($complaint);
    }

    public function update(Complaint $complaint, Request $request)
    {
        $request->validate([
            'status' => 'required'
        ]);

        $complaint->setStatus($request->status);
        $complaint->save();

        return new ComplaintResource($complaint);
    }

    public function destroy(Complaint $complaint)
    {
        Gate::authorize('delete');
        
        $complaint->delete();

        return response()->json([], 204);
    }
}
