<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Project;
use Vtlabs\Gateapp\Models\Resident;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Gateapp\Filters\ResidentFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Http\Resources\Admin\ResidentAdminResource;

class ResidentController extends Controller
{
    public function index(Request $request)
    {
        $residents = Resident::filter($request->all(), ResidentFilter::class)->paginate();

        $projectId = Project::permissionedProject();
        if($projectId) {
            $residents = $residents->where('id', $projectId);
        }

        return ResidentAdminResource::collection($residents);
    }

    public function show(Resident $resident)
    {
        return new ResidentAdminResource($resident);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'type' => 'required',
            'is_approved' => 'required|boolean',
            'project_id' => 'required|exists:gateapp_projects,id',
            'building_id' => 'required|exists:gateapp_buildings,id',
            'flat_id' => 'required|exists:gateapp_flats,id',
            'meta' => 'sometimes|json|nullable',
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta) : null]);
        }

        // create user for resident first  
        $userData = collect($request->only(['name', 'email', 'mobile_number', 'password']));
        $userData = $userData->merge(['mobile_verified' => 1, 'roles' => ['customer']]);
        $user = UserService::create($userData);

        request()->merge(['user_id' => $user->id]);

        $resident = Resident::create($request->only(['type', 'is_approved', 'project_id', 'building_id', 'flat_id', 'user_id']));

        return new ResidentAdminResource($resident);
    }

    public function update(Request $request, Resident $resident)
    {
        $request->validate([
            'type' => 'required',
            'is_approved' => 'required|boolean',
            'project_id' => 'required|exists:gateapp_projects,id',
            'building_id' => 'required|exists:gateapp_buildings,id',
            'flat_id' => 'required|exists:gateapp_flats,id',
            'meta' => 'sometimes|json|nullable',
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta) : null]);
        }

        $resident->fill($request->only(['type', 'is_approved', 'project_id', 'building_id', 'flat_id', 'user_id']));
        $resident->save();

        return new ResidentAdminResource($resident);
    }

    public function destroy(Resident $resident)
    {
        Gate::authorize('delete');

        $resident->delete();

        return response()->json([], 204);
    }
}
