<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Project;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Gateapp\Filters\ProjectFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Http\Resources\Admin\ProjectAdminResource;

class ProjectController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $projects = Project::filter($request->all(), ProjectFilter::class);

        $projectId = Project::permissionedProject();
        if($projectId) {
            $projects = $projects->where('id', $projectId);
        }

        if ($request->pagination == '0') {
            $projects = $projects->get();
        } else {
            $projects = $projects->paginate();
        }

        return ProjectAdminResource::collection($projects);
    }

    public function show(Project $project)
    {
        return new ProjectAdminResource($project);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'address' => 'sometimes',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
            'city_id' => 'required|exists:categories,id',
            'meta' => 'sometimes|json|nullable'
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta) : null]);
        }

        $project = Project::create($request->all());

        return new ProjectAdminResource($project);
    }

    public function update(Request $request, Project $project)
    {
        $request->validate([
            'title' => 'required',
            'address' => 'sometimes',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
            'city_id' => 'required|exists:categories,id',
            'meta' => 'sometimes|json|nullable'
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta) : null]);
        }

        $project->fill($request->all());
        $project->save();

        return new ProjectAdminResource($project);
    }

    public function destroy(Project $project)
    {
        Gate::authorize('delete');
        
        $project->delete();

        return response()->json([], 204);
    }
}
