<?php

namespace Vtlabs\Gateapp\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Gateapp\Models\Project;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Gateapp\Models\Paymentrequest;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Gateapp\Events\PaymentrequestEvent;
use Vtlabs\Gateapp\Filters\PaymentrequestFilter;
use Vtlabs\Gateapp\Http\Resources\Admin\PaymentrequestAdminResource;

class PaymentrequestController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $paymentrequests = Paymentrequest::filter($request->all(), PaymentrequestFilter::class)->latest();

        $projectId = Project::permissionedProject();
        if($projectId) {
            $paymentrequests = $paymentrequests->where('project_id', $projectId);
        }

        return PaymentrequestAdminResource::collection($paymentrequests->paginate());
    }

    public function show(Paymentrequest $paymentrequest)
    {
        return new PaymentrequestAdminResource($paymentrequest);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'amount' => 'required',
            'duedate' => 'required',
            'meta' => 'sometimes|json|nullable',
            'project_id' => 'required|exists:gateapp_projects,id',
            'flat_id' => 'required|exists:gateapp_flats,id',
            'status' => 'required'
        ]);

        request()->merge(['meta' => $request->meta ? json_decode($request->meta) : null]);

        $paymentrequest = Paymentrequest::create($request->only(['title', 'amount', 'duedate', 'meta', 'project_id', 'flat_id']));
        $paymentrequest->setStatus($request->status);
        $paymentrequest->save();

        event(new PaymentrequestEvent($paymentrequest, true));

        return new PaymentrequestAdminResource($paymentrequest);
    }

    public function update(Paymentrequest $paymentrequest, Request $request)
    {
        $request->validate([
            'title' => 'required',
            'amount' => 'required',
            'duedate' => 'required',
            'meta' => 'sometimes|json|nullable',
            'project_id' => 'required|exists:gateapp_projects,id',
            'flat_id' => 'required|exists:gateapp_flats,id',
            'status' => 'required'
        ]);

        request()->merge(['meta' => $request->meta ? json_decode($request->meta) : null]);
        
        $paymentrequest->fill($request->only(['title', 'amount', 'duedate', 'meta', 'project_id', 'flat_id']));
        $paymentrequest->setStatus($request->status);
        $paymentrequest->save();

        return new PaymentrequestAdminResource($paymentrequest);
    }

    public function destroy(Paymentrequest $paymentrequest)
    {
        Gate::authorize('delete');
        
        $paymentrequest->delete();

        return response()->json([], 204);
    }
}
