<?php

namespace Vtlabs\Fantasysport\Services;

use Exception;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Carbon\CarbonPeriod;
use GuzzleHttp\Psr7\Request;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Fantasysport\Models\ApiFootballLeague;
use Vtlabs\Fantasysport\Models\ApiFootballPlayer;
use Vtlabs\Fantasysport\Models\ApiFootballSeason;
use Vtlabs\Fantasysport\Models\ApiFootballFixture;
use Vtlabs\Fantasysport\Models\ApiFootballPlayerEvent;
use Vtlabs\Fantasysport\Models\ApiFootballFixturePlayer;

class FootballApiService
{
    public const API_ENDPOINT = 'https://v3.football.api-sports.io';
    public const API_KEY = '9705d32bf2e5c8b4af433a8cede884c3';
    public const DEFAULT_TIMEOUT = 100;

    // fetch leagues and save in database
    public function fetchLeagues()
    {
        $client = new Client();

        $request = new Request('GET', 'https://v3.football.api-sports.io/leagues', $this->headers());

        $response = $client->send($request, ['timeout' => self::DEFAULT_TIMEOUT]);

        if ($response->getStatusCode() == 200) {
            $response = json_decode($response->getBody()->getContents(), true);
            if (count($response['errors']) > 0) {
                throw new \Exception('Error occured while fetching response');
            }
            foreach ($response['response'] as $league) {
                // create league
                $apiFootballLeague = ApiFootballLeague::firstOrCreate(
                    ['remote_id' => $league['league']['id']],
                    [
                        'name' => $league['league']['name'],
                        'type' => $league['league']['type'],
                        'logo' => $league['league']['logo'],
                        'country_code' => $league['country']['code'],
                        'country_name' => $league['country']['name'],
                        'country_flag' => $league['country']['flag']
                    ]
                );

                foreach ($league['seasons'] as $season) {
                    $leagueRemoteId = $league['league']['id'];

                    // get if season already exists in database
                    $apiFootballSeason = ApiFootballSeason::whereHas('league', function ($query) use ($leagueRemoteId) {
                        return $query->where('remote_id', $leagueRemoteId);
                    })->where('year', $season['year'])->first();

                    if (!$apiFootballSeason) {
                        $apiFootballSeason = new ApiFootballSeason();
                    }
                    $apiFootballSeason->fill([
                        'year' => $season['year'],
                        'start' => $season['start'],
                        'end' => $season['end'],
                        'coverage_fixtures_event' => $season['coverage']['fixtures']['events'],
                        'coverage_fixtures_lineups' => $season['coverage']['fixtures']['lineups'],
                        'coverage_fixtures_statistics_fixtures' => $season['coverage']['fixtures']['statistics_fixtures'],
                        'coverage_fixtures_statistics_players' => $season['coverage']['fixtures']['statistics_players'],
                        'coverage_standings' => $season['coverage']['standings'],
                        'coverage_players' => $season['coverage']['players'],
                        'coverage_top_scorers' => $season['coverage']['top_scorers'],
                        'coverage_top_assists' => $season['coverage']['top_assists'],
                        'coverage_top_cards' => $season['coverage']['top_cards'],
                        'coverage_injuries' => $season['coverage']['injuries'],
                        'coverage_predictions' => $season['coverage']['predictions'],
                        'coverage_odds' => $season['coverage']['odds'],
                        'league_id' => $apiFootballLeague->id
                    ]);
                    $apiFootballSeason->save();
                }

                // after saving leagues we need to filter out those leagues for which appropriate data is not available

                // first get the latest season
                $latestSeason = $apiFootballLeague->seasons()->orderByDesc('year')->first();

                // decide if we can keep the league
                if (!$latestSeason->coverage_fixtures_event || !$latestSeason->coverage_fixtures_lineups) {
                    $apiFootballLeague->delete();
                }
            }
        }
    }

    // fetch fixtures and save in database
    public function fetchFixtures()
    {
        // get enabled leagues for which we will filter fixtures
        $leagues = ApiFootballLeague::where('enabled', 1)->get();

        // fetch fixtures of next x days
        $period = CarbonPeriod::create(Carbon::now(), Carbon::now()->addDays(14)); // @TODO: get number of days from settings

        // Iterate over the period
        foreach ($period as $date) {
            $dateFormatted = $date->format('Y-m-d');

            foreach ($leagues as $league) {
                $client = new Client();

                // get latest season of the league
                $season = ApiFootballSeason::where('league_id', $league['id'])->orderByDesc('year')->first();

                $params = ['date' => $dateFormatted, 'league' => $league['remote_id'], 'season' => $season['year'], 'status' => 'NS'];
                //$params = ['id' => '720927'];

                $request = new Request('GET', 'https://v3.football.api-sports.io/fixtures?' .  http_build_query($params), $this->headers());

                $response = $client->send($request, ['timeout' => self::DEFAULT_TIMEOUT]);

                //sleep(7);

                if ($response->getStatusCode() == 200) {
                    $response = json_decode($response->getBody()->getContents(), true);
                    if (count($response['errors']) > 0) {
                        throw new \Exception('Error occured while fetching response' . json_encode($response['errors']));
                    }
                    foreach ($response['response'] as $fixture) {
                        $apiFootballFixture = ApiFootballFixture::where('remote_id',  $fixture['fixture']['id'])->first();

                        if (!$apiFootballFixture) {
                            $apiFootballFixture = ApiFootballFixture::firstOrCreate(
                                ['remote_id' => $fixture['fixture']['id']],
                                [
                                    'timezone' => $fixture['fixture']['timezone'],
                                    'date' => Carbon::createFromFormat('Y-m-d\TH:i:sP', $fixture['fixture']['date'])->format('Y-m-d H:i:s'),

                                    'venue_name' => $fixture['fixture']['venue']['name'],
                                    'venue_city' => $fixture['fixture']['venue']['city'],

                                    'status' => $fixture['fixture']['status']['short'],
                                    'status_long' => $fixture['fixture']['status']['long'],

                                    'remote_league_id' => $fixture['league']['id'],
                                    'league_name' => $fixture['league']['name'],
                                    'league_country' => $fixture['league']['country'],
                                    'league_logo' => $fixture['league']['logo'],
                                    'league_flag' => $fixture['league']['flag'],
                                    'season' => $fixture['league']['season'],
                                    'round' => $fixture['league']['round'],

                                    'remote_home_team_id' => $fixture['teams']['home']['id'],
                                    'home_team_name' => $fixture['teams']['home']['name'],
                                    'home_team_logo' => $fixture['teams']['home']['logo'],
                                    'remote_away_team_id' => $fixture['teams']['away']['id'],
                                    'away_team_name' => $fixture['teams']['away']['name'],
                                    'away_team_logo' => $fixture['teams']['away']['logo'],
                                ]
                            );

                            // fetch players of teams
                            $this->fetchPlayerByTeam($fixture['teams']['home']['id'], $season['year']);
                            sleep(60);
                            $this->fetchPlayerByTeam($fixture['teams']['away']['id'], $season['year']);
                            sleep(60);
                        }
                    }
                }
            }
            sleep(10);
        }
    }

    public function fetchLineUp($fixtureId = null)
    {
        if (!$fixtureId) {
            $apiFootballFixtures = ApiFootballFixture::where('enabled', 1)->get();

            foreach ($apiFootballFixtures as $apiFootballFixture) {
                $this->fetchFixtureLineup($apiFootballFixture);
            }
        } else {
            $this->fetchFixtureLineup(ApiFootballFixture::find($fixtureId));
        }
    }

    public function fetchEvents($fixtureId)
    {
        $apiFootballFixture = ApiFootballFixture::find($fixtureId);

        $client = new Client();

        $params = ['fixture' => $apiFootballFixture->remote_id];

        $request = new Request('GET', 'https://v3.football.api-sports.io/fixtures/events?' .  http_build_query($params), $this->headers());

        $response = $client->send($request, ['timeout' => self::DEFAULT_TIMEOUT]);

        $settings = CoreHelper::settingsAsDictionary();

        if ($response->getStatusCode() == 200) {
            $response = json_decode($response->getBody()->getContents(), true);
            if (count($response['errors']) > 0) {
                throw new \Exception('Error occured while fetching response' . json_encode($response['errors']));
            }

            foreach ($response['response'] as $event) {

                $eventKey = $this->eventKey($event['type'], $event['detail']);

                if ($eventKey) {
                    $settingKey = 'points_' . $eventKey;
                    $points = $settings[$settingKey];
                    $hash = $this->generateEventHash($event);

                    if (ApiFootballPlayerEvent::where('hash', $hash)->count() == 0) {
                        ApiFootballPlayerEvent::create(
                            [
                                'type' => $eventKey,
                                'points' => intval($points),
                                'fixture_id' => $fixtureId,
                                'hash' => $hash,
                                'remote_player_id' => $event['player']['id']
                            ]
                        );
                    }
                }
            }
        }
    }

    public function updateFixture($fixtureId)
    {
        $apiFootballFixture = ApiFootballFixture::find($fixtureId);

        $client = new Client();

        $params = ['id' => $apiFootballFixture->remote_id];

        $request = new Request('GET', 'https://v3.football.api-sports.io/fixtures?' .  http_build_query($params), $this->headers());

        $response = $client->send($request, ['timeout' => self::DEFAULT_TIMEOUT]);

        if ($response->getStatusCode() == 200) {
            $response = json_decode($response->getBody()->getContents(), true);

            $fixture = $response['response'][0]['fixture'];

            if ($fixture) {

                if ($apiFootballFixture->status != $fixture['status']['short']) {
                    $apiFootballFixture->status = $fixture['status']['short'];
                    $apiFootballFixture->status_long = $fixture['status']['long'];

                    if ($fixture['status']['short'] == 'FT') {
                        // @todo - create listeners

                        // calculate each player's point
                        $playerPoints = [];
                        $fixtureEvents = ApiFootballPlayerEvent::where('fixture_id', $fixtureId)->get();
                        foreach ($fixtureEvents as $fixtureEvent) {
                            $playerPoints[$fixtureEvent['remote_player_id']] += $fixtureEvent['points'];
                        }

                        foreach ($playerPoints as $remotePlayerId => $points) {
                            $fixturePlayer = ApiFootballFixturePlayer::where('remote_player_id', $remotePlayerId)->first();
                            if ($fixturePlayer) {
                                $fixturePlayer->points = $points >= 0 ? $points : 0;
                                $fixturePlayer->save();
                            }
                        }

                        // @todo - update teams point as well
                    }

                    $apiFootballFixture->save();
                }
            }
        }
    }

    private function fetchPlayerByTeam($remoteTeamId, $season, $page = 1)
    {
        $client = new Client();

        $params = ['team' => $remoteTeamId, 'season' => $season, 'page' => $page];

        $request = new Request('GET', 'https://v3.football.api-sports.io/players?' .  http_build_query($params), $this->headers());

        $response = $client->send($request, ['timeout' => self::DEFAULT_TIMEOUT]);

        if ($response->getStatusCode() == 200) {
            $response = json_decode($response->getBody()->getContents(), true);
            if (count($response['errors']) > 0) {
                throw new \Exception('Error occured while fetching response' . json_encode($response['errors']));
            }
            foreach ($response['response'] as $player) {
                ApiFootballPlayer::firstOrCreate(
                    ['remote_id' => $player['player']['id']],
                    [
                        'name' => $player['player']['name'],
                        'firstname' => $player['player']['firstname'],
                        'lastname' => $player['player']['lastname'],
                        'age' => $player['player']['age'],
                        'nationality' => $player['player']['nationality'],
                        'height' => $player['player']['height'],
                        'weight' => $player['player']['weight'],
                        'photo' => $player['player']['photo'],
                    ]
                );
            }

            // pagination
            if ($response['paging']['current'] !=  $response['paging']['total']) {
                $this->fetchPlayerByTeam($remoteTeamId, $season, $response['paging']['current'] + 1);
            }
        }
    }

    private function fetchFixtureLineup($apiFootballFixture)
    {
        $client = new Client();

        $params = ['fixture' => $apiFootballFixture->remote_id];

        $request = new Request('GET', 'https://v3.football.api-sports.io/fixtures/lineups?' .  http_build_query($params), $this->headers());

        $response = $client->send($request, ['timeout' => self::DEFAULT_TIMEOUT]);

        if ($response->getStatusCode() == 200) {
            $response = json_decode($response->getBody()->getContents(), true);
            if (count($response['errors']) > 0) {
                throw new \Exception('Error occured while fetching response' . json_encode($response['errors']));
            }
            foreach ($response['response'] as $team) {
                // iterate over playing XI
                $players = array_merge($team['startXI'], $team['substitutes']);
                foreach ($players as $player) {
                    if (!$player['player']['id']) continue;

                    ApiFootballFixturePlayer::create([
                        'remote_team_id' => $team['team']['id'],
                        'remote_player_id' => $player['player']['id'],
                        'name' => $player['player']['name'],
                        'number' => $player['player']['number'],
                        'pos' => $player['player']['pos'],
                        'grid' => $player['player']['grid'],
                        'fixture_id' => $apiFootballFixture->id
                    ]);
                }
            }
        }
    }

    private function headers()
    {
        return array(
            'x-rapidapi-host' => 'v3.football.api-sports.io',
            'x-rapidapi-key' => self::API_KEY,
            'Accept' => 'application/json'
        );
    }

    private function eventKey($type, $detail)
    {
        $key = null;

        if ($type == 'Goal') {
            if ($detail == 'Normal Goal') {
                $key = 'goal_normal';
            }
            if ($detail == 'Own Goal') {
                $key = 'goal_own';
            }
            if ($detail == 'Penalty') {
                $key = 'goal_penalty';
            }
            if ($detail == 'Missed Goal') {
                $key = 'goal_missed_penalty';
            }
        }

        if ($type == 'Card') {
            if ($detail == 'Yellow Card') {
                $key = 'card_yellow';
            }

            if ($detail == 'Second Yellow Card') {
                $key = 'card_yellow_second';
            }

            if ($detail == 'Red Card') {
                $key = 'card_red';
            }
        }

        return $key;
    }

    private function generateEventHash($event)
    {
        $time = intval($event['time']['elapsed']) . '-' . intval($event['time']['extra']);
        return md5($event['time']['elapsed']);
    }
}
