<?php

namespace Vtlabs\Fantasysport\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Fantasysport\Models\Contest;
use Vtlabs\Fantasysport\Models\ContestTeam;
use Vtlabs\Core\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use Vtlabs\Fantasysport\Filters\ContestFilter;
use Vtlabs\Fantasysport\Http\Resources\ContestLeaderboardResource;
use Vtlabs\Fantasysport\Http\Resources\ContestResource;
use Vtlabs\Fantasysport\Http\Resources\ContestWinningResource;

class ContestController extends Controller
{
    public function __construct()
    {
        if (array_key_exists('HTTP_AUTHORIZATION', $_SERVER) && $_SERVER['HTTP_AUTHORIZATION']) {
            $this->middleware('auth:api');
        }
    }

    public function index(Request $request)
    {
        $request->validate([]);

        $contests = Contest::filter($request->all(), ContestFilter::class);

        return ContestResource::collection($contests->paginate());
    }

    public function joinContest(Contest $contest, Request $request)
    {
        $request->validate([
            'team_id' => 'required',
            'meta' => 'sometimes'
        ]);

        $request->merge([
            "meta" => $request->meta ? json_decode($request->meta) : null
        ]);

        // @TODO validations before joining a team such as spots left, multiple entries etc

        if ($contest->entry_amount > Auth::user()->balance) {
            throw ValidationException::withMessages(['message' => 'User does not have enough balance']);
        }

        ContestTeam::create([
            'meta' => $request->meta,
            'contest_id' => $contest->id,
            'team_id' => $request->team_id
        ]);

        return response()->json(((object)[]));
    }

    public function winnings(Contest $contest, Request $request)
    {
        $winnings = $contest->winnings()->get();

        return ContestWinningResource::collection($winnings);
    }

    public function leaderboard(Request $request)
    {
        $leaderboard = DB::select('SELECT team_id, points, FIND_IN_SET( points, (SELECT GROUP_CONCAT( DISTINCT points ORDER BY points DESC ) FROM fsport_contest_teams) ) AS leaderboard_rank FROM fsport_contest_teams;');

        return ContestLeaderboardResource::collection($leaderboard);
    }
}
