<?php

namespace Vtlabs\Event\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Event\Models\Venue;
use Illuminate\Support\Facades\DB;
use Vtlabs\Event\Models\Auditorium;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Event\Filters\VenueFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Event\Http\Resources\Admin\VenueAdminResource;

class VenueController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $venues = Venue::filter($request->all(), VenueFilter::class);

        if ($request->pagination == '0') {
            $venues = $venues->get();
        } else {
            $venues = $venues->paginate();
        }

        return VenueAdminResource::collection($venues);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'short_description_translations' => 'required|translation',
            'description_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'area' => 'sometimes|string',
            'address' => 'sometimes|string',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "short_description" => request()->input('_short_description'),
            "description" => request()->input('_description')
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $venue = Venue::create($request->only(['title', 'short_description', 'description', 'meta', 'area', 'address', 'longitude', 'latitude']));

        return new VenueAdminResource($venue->fresh());
    }

    public function show(Venue $venue)
    {
        return new VenueAdminResource($venue);
    }

    public function update(Venue $venue, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'short_description_translations' => 'required|translation',
            'description_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'area' => 'sometimes|string',
            'address' => 'sometimes|string',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "short_description" => request()->input('_short_description'),
            "description" => request()->input('_description')
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $venue->fill($request->only(['title', 'short_description', 'description', 'meta', 'area', 'address', 'longitude', 'latitude']));
        $venue->save();

        // update events location when venues locations are updated
        $auditoriums = Auditorium::where('venue_id', $venue->id)->get()->pluck(['id'])->toArray();
        DB::table('ev_events')->whereIn('auditorium_id', $auditoriums)->update([
            'address' => $venue->address,
            'longitude' => $venue->longitude,
            'latitude' => $venue->latitude
        ]);

        return new VenueAdminResource($venue->fresh());
    }

    public function destroy(Venue $venue)
    {
        Gate::authorize('delete');

        $venue->delete();

        return response()->json([], 200);
    }
}
