<?php

namespace Vtlabs\Event\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Event\Models\Artist;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Event\Filters\ArtistFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Event\Http\Resources\Admin\ArtistAdminResource;

class ArtistController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $artists = Artist::filter($request->all(), ArtistFilter::class);

        if ($request->pagination == '0') {
            $artists = $artists->get();
        } else {
            $artists = $artists->paginate();
        }

        return ArtistAdminResource::collection($artists);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'short_description_translations' => 'required|translation',
            'description_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'images.*' => 'sometimes|image',
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "short_description" => request()->input('_short_description'),
            "description" => request()->input('_description')
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $artist = Artist::create($request->only(['title', 'short_description', 'description', 'meta']));

        // images
        if (!empty($request->images)) {
            foreach ($request->images as $image) {
                $artist->addMedia($image)->toMediaCollection("images");
            }
        }

        return new ArtistAdminResource($artist->fresh());
    }

    public function show(Artist $artist)
    {
        return new ArtistAdminResource($artist);
    }

    public function update(Artist $artist, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'short_description_translations' => 'required|translation',
            'description_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'images.*' => 'sometimes|image',
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "short_description" => request()->input('_short_description'),
            "description" => request()->input('_description')
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $artist->fill($request->only(['title', 'short_description', 'description', 'meta']));
        $artist->save();

        // images
        if (!empty($request->images)) {
            $artist->clearMediaCollection('images');
            foreach ($request->images as $image) {
                $artist->addMedia($image)->toMediaCollection("images");
            }
        }

        return new ArtistAdminResource($artist->fresh());
    }

    public function destroy(Artist $artist)
    {
        Gate::authorize('delete');

        $artist->delete();

        return response()->json([], 200);
    }
}
