<?php

namespace Vtlabs\Event\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Event\Models\Seat;
use Vtlabs\Event\Models\Event;
use Vtlabs\Event\Models\OrderSeat;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Ecommerce\Models\Coupon;
use Vtlabs\Event\Models\EventOrder;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Event\Filters\EventOrderFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use Vtlabs\Event\Http\Resources\EventOrderResource;

/**
 * @group  Event
 *
 * APIs for event order
 */
class EventOrderController extends Controller
{
    private $settings;

    public function __construct()
    {
        $this->settings = CoreHelper::settingsAsDictionary();
    }

    public function index(Request $request)
    {
        $request->validate([
            'paid' => 'sometimes'
        ]);

        $request->merge([
            'user' => Auth::id()
        ]);

        $eventOrders = EventOrder::filter($request->all(), EventOrderFilter::class)->ordered();

        return EventOrderResource::collection($eventOrders->paginate());
    }

    /**
     * Purchase event tickets
     */
    public function store(Event $event, Request $request)
    {
        $request->validate([
            'seats' => 'required|array|exists:ev_seats,id',
            'coupon_code' => 'sometimes',
            'meta' => 'sometimes|json',
            'payment_method_slug' => 'required|exists:payment_methods,slug',
        ]);

        // check if seat is not already booked
        $alreadyBooked = OrderSeat::whereIn('seat_id', $request->seats)->whereHas('order.event', function ($query) use ($event) {
            $query->where('id', $event->id);
        })->exists();

        if($alreadyBooked) {
            throw ValidationException::withMessages(['seats' => 'Already booked']);
        }

        // get price of each selected seat
        $subtotal = 0;
        foreach($request->seats as $seatId) {
            $seat = Seat::find($seatId);

            foreach($event->eventCategories as $eventCategory) {
                if($seat->category_id == $eventCategory->pivot->category_id) {
                    $subtotal += $eventCategory->pivot->price;
                }
            }
        }

        $taxes = $this->calculateTax($subtotal);
        $discount = $this->calculateDiscount($request->coupon_code, $subtotal);
        $total = ($subtotal + $taxes) - $discount;

        // create an order instance
        $order = EventOrder::create([
            'subtotal' => $subtotal,
            'taxes' => $taxes,
            'discount' => $discount,
            'total' => $total,
            'user_id' => Auth::id(),
            'event_id' => $event->id,
            'meta' => !empty($request->meta) ? json_decode($request->meta) : null
        ]);

        $order->seats()->attach($request->seats);

        // payment
        $paymentMethodSlug = $request->payment_method_slug;
        $payment = Auth::user()->createPayment($order, $total, $paymentMethodSlug);

        // if event is free, mark the payment as paid
        if($subtotal == 0) {
            $payment->setStatus('paid');
            $payment->save();
        }

        // allot point to member
        if($total > 0 && Auth::user()->hasActiveSubscription()) {
            $user = Auth::user();
            $settings = CoreHelper::settingsAsDictionary();
            $meta = $user->meta;
            $meta['points'] = floatval($meta['points']) + ($total / floatval($settings['points_allotment_rate']));
            $user->meta = $meta;
            $user->save();
        }

        return new EventOrderResource($order->fresh());
    }

    public function update(EventOrder $eventOrder, Request $request)
    {
        $request->validate([
            'meta' => 'sometimes|json'
        ]);

        if($request->meta) {
            $eventOrder->meta = json_decode($request->meta);
        }

        return new EventOrderResource($eventOrder->fresh());
    }

    private function calculateTax($amount)
    {
        $tax = (float )$this->settings['tax_in_percent'] ?? 0;
        return ($amount * $tax) / 100;
    }

    private function calculateDiscount($coupon_code, $amount)
    {
        $discount = 0;

        if (!isset($coupon_code)) {
            return 0;
        }

        try {
            $coupon = Coupon::getCouponFromCode($coupon_code, $this->user);
            if ($coupon !== null) {
                if ($coupon->type == 'fixed') {
                    $discount = $coupon->reward;
                }

                if ($coupon->type == 'percent') {
                    $discount = ($amount * $coupon->reward) / 100;
                }

                $coupon->users()->attach($this->user->id, [
                    'used_at' => Carbon::now(),
                ]);
            }
        } catch (\Exception $ex) {
            //
        }

        return $discount;
    }
}
