<?php

namespace Vtlabs\Event\Http\Controllers\Api;

use Illuminate\Http\Request;
use Vtlabs\Event\Models\Seat;
use Vtlabs\Event\Models\Event;
use Vtlabs\Core\Models\User\User;
use Vtlabs\Event\Models\OrderSeat;
use Vtlabs\Event\Models\EventOrder;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Event\Filters\EventFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Core\Http\Resources\UserResource;
use Vtlabs\Event\Http\Resources\SeatResource;
use Vtlabs\Event\Http\Resources\EventResource;
use ChristianKuri\LaravelFavorite\Models\Favorite;
use Vtlabs\Category\Http\Resources\CategoryResource;
use Vtlabs\Event\Http\Resources\EventOrderSeatResource;

/**
 * @group Event
 *
 * APIs for event requests
 */
class EventController extends Controller
{
    public function __construct()
    {
        if (array_key_exists('HTTP_AUTHORIZATION', $_SERVER)) {
            $this->middleware('auth:api');
        }
    }

    public function index(Request $request)
    {
        $request->validate([
            'lat' => 'sometimes|numeric',
            'long' => 'sometimes|numeric',
            'search' => 'sometimes',
            'sort' => 'sometimes|in:distance,default',
            'artist' => 'sometimes|exists:ev_artists,id'
        ]);

        $events = Event::filter($request->all(), EventFilter::class);

        return EventResource::collection($events->paginate());
    }

    public function seats(Event $event, Request $request)
    {
        // all the categories this event's auditoriums seats have
        $categories = [];
        $distinctCategoriesOfSeats =  $event->auditorium->seats->unique('category_id');
        foreach ($distinctCategoriesOfSeats as $seat) {
            $categories[] = $seat->category;
        }

        // all the seats in this auditorium
        $auditoriumSeats = $event->auditorium->seats;

        // in each seat add a field is_booked indicating whether seat is booked or not
        foreach ($auditoriumSeats as $seat) {
            $eventId = $event->id;
            $seat->is_booked = OrderSeat::where('seat_id', $seat->id)->whereHas('order.event', function ($query) use ($eventId) {
                $query->where('id', $eventId);
            })->exists();
        }

        return [
            "categories" => CategoryResource::collection($categories),
            "seats" => SeatResource::collection($auditoriumSeats)
        ];
    }

    public function favourites(Request $request)
    {
        $eventList = Auth::user()->favorite(Event::class)->flatten();

        return EventResource::collection($eventList);
    }

    public function favoriters(Event $event)
    {
        $userIds = Favorite::where('favoriteable_type', Event::class)->get()->pluck(['user_id']);

        $users = User::whereIn('id', $userIds)->paginate();

        return UserResource::collection($users);
    }

    /**
     * Toggle Favourite
     * 
     * @authenticated
     * 
     * @response []
     */
    public function toggleFavourite(Event $event, Request $request)
    {
        $event->toggleFavorite();

        return response()->json((object)[]);
    }

    public function usersAttending(Event $event, Request $request)
    {
        // we return at max 15 users in this api, total number of users attending the event can be get from event object
        $userIds = EventOrder::where('event_id', $event->id)->select('user_id')->distinct()->limit(15)->get()->pluck(['user_id'])->toArray();

        $users = User::whereIn('id', $userIds)->get();

        return UserResource::collection($users);
    }

    public function getUnbookedStandingSeats(Event $event, Request $request)
    {
        $request->validate([
            'seats' => 'required|numeric'
        ]);
        $allStandingSeats = Seat::where('auditorium_id', $event->auditorium_id)->where('type', 'standing')->pluck('id')->toArray();
        $bookedSeats = $event->seats()->pluck('seat_id')->toArray();

        $remainingSeats = array_diff($allStandingSeats, $bookedSeats);

        return response()->json(['seats' => array_slice($remainingSeats, 0, $request->seats)]);
    }
}
