<?php

namespace Vtlabs\Event\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Event\Models\Partner;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Event\Filters\PartnerFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Event\Http\Resources\Admin\PartnerAdminResource;

class PartnerController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string'
        ]);

        $partners = Partner::filter($request->all(), PartnerFilter::class)->latest();

        if ($request->pagination == '0') {
            $partners = $partners->get();
        } else {
            $partners = $partners->paginate();
        }

        return PartnerAdminResource::collection($partners);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'images.*' => 'sometimes|image',
            'area' => 'sometimes|string',
            'address' => 'sometimes|string',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90',
        ]);

        CoreHelper::requestMergeJsonField();

        $partner = Partner::create($request->only(['title', 'description','meta', 'area', 'address', 'longitude', 'latitude']));

        
        // images
        if (!empty($request->images)) {
            foreach ($request->images as $image) {
                $partner->addMedia($image)->toMediaCollection("images");
            }
        }

        return new PartnerAdminResource($partner->fresh());
    }

    public function show(Partner $partner)
    {
        return new PartnerAdminResource($partner);
    }

    public function update(Partner $partner, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'images.*' => 'sometimes|image',
            'area' => 'sometimes|string',
            'address' => 'sometimes|string',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90'
        ]);

        CoreHelper::requestMergeJsonField();

        $partner->fill($request->only(['title', 'description','meta', 'area', 'address', 'longitude', 'latitude']));
        $partner->save();

        // images
        if (!empty($request->images)) {
            $partner->clearMediaCollection('images');
            foreach ($request->images as $image) {
                $partner->addMedia($image)->toMediaCollection("images");
            }
        }

        return new PartnerAdminResource($partner->fresh());
    }

    public function destroy(Partner $partner)
    {
        Gate::authorize('delete');

        $partner->delete();

        return response()->json([], 200);
    }
}
